#!/bin/bash

# Copyright (c) 2003-2021 Broadcom. All Rights Reserved.
# The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.

SUCCESS=0
FAILURE=1
STATUS=$SUCCESS
KITNAME=""
PRODUCT_STRING="Emulex HBA Manager"
LOGFILE_NAME="utils-install.log"
FHS_SUPPORT=0

# Directory paths
INSTALL_DIR="usr/sbin/ocmanager"
LOG_DIR="var/opt/emulex/ocmanager/logs"

# EHM scripts
STOP_EHM_SH="/${INSTALL_DIR}/stop_ocmanager"

# Uninstallation log
LOGFILE="/${INSTALL_DIR}/${LOGFILE_NAME}"

# Function:  log_message()
#
# Description:
#   Print the strings passed as parameters to both STDOUT as well as
#   to a log file.
#
# Parameters:
#   A series of strings.
#
# Returns:
#   0 on success.
log_message()
{
	local ECHO_OPTIONS
	LOGDIRECTORY=${LOGFILE%/*}
	if [[ ! -d ${LOGDIRECTORY} ]] ; then
		mkdir -p ${LOGDIRECTORY}
	fi

	if [[ $# -eq 0 ]] ; then
		echo "" | tee -a ${LOGFILE}
	elif [[ -z "$1" ]] ; then
		echo "" | tee -a ${LOGFILE}
	elif [[ "$1" = "-init" ]] ; then
		echo "$0 - $(date)" >> ${LOGFILE}
	else
		while [[ -n "$1" ]] ; do
			if [[ "${1:0:1}" = "-" ]] ; then
				ECHO_OPTIONS="${1}"
			else
				echo ${ECHO_OPTIONS} "$1" | tee -a ${LOGFILE}
			fi
			shift
		done
	fi
	return $SUCCESS
}

fhsSupport()
{
	# Determine OS
	if [[ -f /etc/redhat-release ]]; then
		grep -q Citrix /etc/issue || grep -q XenServer /etc/issue
		if [[ $? -eq 0 ]]; then
			return
		fi

		oracleos=0
		if [[ -f /etc/oracle-release ]]; then
			oracleos=`grep Oracle /etc/oracle-release | wc -l`
		fi

		if [[ $oracleos -eq 1 ]]; then
			RELEASE_FILE="/etc/oracle-release"
		else
			RELEASE_FILE="/etc/redhat-release"
		fi

		major_version=`cat $RELEASE_FILE | awk -F'.' '{print $1}' | awk '{print $NF}'`
		minor_version=`cat $RELEASE_FILE | awk -F'.' '{print $2}' | cut -d' ' -f1`
		if [[ "$major_version" -eq 9 ]]; then
			FHS_SUPPORT=1
		fi
	fi

	if [[ $FHS_SUPPORT -eq 1 ]]; then
		INSTALL_DIR="opt/emulex/ocmanager"
		STOP_EHM_SH="/${INSTALL_DIR}/scripts/stop_ocmanager.sh"
		LOGFILE="/${LOG_DIR}/${LOGFILE_NAME}"
	fi
}

initLog()
{
	log_message -init
}

stop_ocmanager()
{
	if [[ -f ${STOP_OCM_SH} ]]; then
		echo "${STOP_OCM_SH}" >> ${LOGFILE}
		${STOP_OCM_SH} >> ${LOGFILE} 2>&1
	fi
}

uninstallOcVisionSensor()
{
	rpm -q --quiet OneCommandVisionSensor
	if [[ $? -eq 0 ]]; then
		SENSOR_RPM=`rpm -q OneCommandVisionSensor`
		log_message "Warning: sensor component will be un-installed as it is dependent on OCM."
		log_message "Do you want to proceed?"
		log_message "Enter y to uninstall both OCM and OCV sensor. "
		log_message "Enter n to keep OCM and OCV sensor. "
		log_message -n "Enter the letter 'y' or 'n'"
		read uninstall_sensor
		if [[ "$uninstall_sensor" == "Y" || "$uninstall_sensor" == "y" ]]; then
			log_message "Uninstalling OneCommand Sensor and $PRODUCT_STRING kits."

			if [[ -f /${INSTALL_DIR}/scripts/OneCommandVisionSensor*/uninstall-sensor ]]; then
				/${INSTALL_DIR}/scripts/OneCommandVisionSensor*/uninstall-sensor
			else
				echo "Uninstalling OneCommandVisionSensor: $SENSOR_RPM"
				rpm -ev $SENSOR_RPM
			fi

			SENSORSTATUS=$?
			if [[ "$SENSORSTATUS" = "0" ]]; then
				log_message "Uninstalling OneCommandVisionSensor completed successfully"
			else
				log_message "An error occured while uninstalling OneCommandVisionSensor"
				STATUS=$FAILURE
			fi
		else
			exit  $SUCCESS
		fi
		log_message ""
	fi
}

uninstallElxGuiKit()
{
	rpm -q --quiet elxocmguikit
	if [[ $? -eq 0 ]]; then
		HBAGUIQUERY=`rpm -q elxocmguikit`
		KITNAME="Enterprise"
		log_message "Uninstalling elxocmguikit: $HBAGUIQUERY"
		rpm -e $HBAGUIQUERY
		GUISTATUS=$?
		if [[ "$GUISTATUS" = "0" ]]; then
			log_message "Uninstalling $HBAGUIQUERY completed successfully"
		else
			log_message "An error occurred while uninstalling $HBAGUIQUERY"
			STATUS=$FAILURE
		fi
		log_message ""
	fi
}

uninstallElxCoreKit()
{
	rpm -q --quiet elxocmcorekit
	if [[ $? -eq 0 ]]; then
		if [[ -z $KITNAME ]]; then
			KITNAME="Core"
		fi
		HBACOREQUERY=`rpm -q elxocmcorekit`
		log_message "Uninstalling elxocmcorekit: $HBACOREQUERY"
		rpm -e $HBACOREQUERY
		CORESTATUS=$?
		if [[ "$CORESTATUS" = "0" ]]; then
			log_message "Uninstalling $HBACOREQUERY completed successfully"
		else
			log_message "An error occurred while uninstalling $HBACOREQUERY"
			STATUS=$FAILURE
		fi
		log_message ""
	fi
}

uninstallElxGUI()
{
	rpm -q --quiet elxocmgui
	if [[ $? -eq 0 ]]; then
		HBAGUIQUERY=`rpm -q elxocmgui`
		KITNAME="Enterprise"
		log_message "Uninstalling elxocmgui: $HBAGUIQUERY"
		rpm -e $HBAGUIQUERY
		GUISTATUS=$?
		if [[ "$GUISTATUS" = "0" ]]; then
			log_message "Uninstalling $HBAGUIQUERY completed successfully"
		else
			log_message "An error occurred while uninstalling $HBAGUIQUERY"
			STATUS=$FAILURE
		fi
		log_message ""
	fi
}

uninstallElxJVM()
{
	rpm -q --quiet elxocmjvm
	if [[ $? -eq 0 ]]; then
		HBAJVMQUERY=`rpm -q elxocmjvm`
		KITNAME="Enterprise"
		log_message "Uninstalling elxocmjvm: $HBAJVMQUERY"
		rpm -e $HBAJVMQUERY
		JVMSTATUS=$?
		if [[ "$JVMSTATUS" = "0" ]]; then
			log_message "Uninstalling $HBAJVMQUERY completed successfully"
		else
			log_message "An error occurred while uninstalling $HBAJVMQUERY"
			STATUS=$FAILURE
		fi
		log_message ""
	fi
}

uninstallElxOcmCore()
{
	rpm -q --quiet elxocmcore
	if [[ $? -eq 0 ]]; then
		if [[ -z $KITNAME ]]; then
			KITNAME="Core"
		fi
		HBACOREQUERY=`rpm -q elxocmcore`
		log_message "Uninstalling elxocmcore: $HBACOREQUERY"
		rpm -e $HBACOREQUERY
		CORESTATUS=$?
		if [[ "$CORESTATUS" = "0" ]]; then
			log_message "Uninstalling $HBACOREQUERY completed successfully"
		else
			log_message "An error occurred while uninstalling $HBACOREQUERY"
			STATUS=$FAILURE
		fi
		log_message ""
	fi
}

uninstallElxOcmCorelibs()
{
	rpm -q --quiet elxocmcorelibs
	if [[ $? -eq 0 ]]; then
		if [[ -z $KITNAME ]]; then
			KITNAME="Core"
		fi
		HBACORELIBSQUERY=`rpm -q elxocmcorelibs`
		log_message "Uninstalling elxocmcorelibs: $HBACORELIBSQUERY"
		rpm -e $HBACORELIBSQUERY
		CORELIBSSTATUS=$?
		if [[ "$CORELIBSSTATUS" = "0" ]]; then
			log_message "Uninstalling $HBACORELIBSQUERY completed successfully"
		else
			log_message "An error occurred while uninstalling $HBACORELIBSQUERY"
			STATUS=$FAILURE
		fi
		log_message ""
	fi
}

uninstallEnablementKit()
{
	echo $* | grep -q "all-libs"
	if [[ $? -eq 0 ]]; then
		for enablement_pkg in \
			HP-CNA-FC-Emulex-Enablement-Kit       \
			HP-CNA-FC-mezz-Emulex-Enablement-Kit  \
			HP-CNA-FC-sy-Emulex-Enablement-Kit    \
			HPE-CNA-FC-Emulex-Enablement-Kit      \
			HPE-CNA-FC-mezz-Emulex-Enablement-Kit \
			HPE-CNA-FC-sy-Emulex-Enablement-Kit; do

			rpm -q --quiet $enablement_pkg
			if [[ $? -eq 0 ]]; then
				if [[ -z $KITNAME ]]; then
					KITNAME="Core"
				fi
				HBAHPCORELIBSQUERY=$(rpm -q $enablement_pkg)
				log_message "Uninstalling $enablement_pkg: $HBAHPCORELIBSQUERY"
				rpm -e $HBAHPCORELIBSQUERY
				CORELIBSSTATUS=$?
				if [[ "$CORELIBSSTATUS" = "0" ]]; then
					log_message "Uninstalling $HBAHPCORELIBSQUERY completed successfully"
				else
					log_message "An error occurred while uninstalling $HBAHPCORELIBSQUERY"
					STATUS=$FAILURE
				fi
				log_message ""
		fi
		done
	fi
}

uninstallHbaapiWrapper()
{
	HBAAPI_LIST=`rpm -qa | grep hbaapiwrapper`
	if [[ $? -eq 0 ]]; then
		if [[ -z $KITNAME ]]; then
			KITNAME="Core"
		fi
		rpm -e --quiet --test hbaapiwrapper > /dev/null 2>&1
		if [[ $? -eq 0 ]]; then
			for hbaapirpm in $HBAAPI_LIST
			do
				log_message "Uninstalling hbaapiwrapper: $hbaapirpm"
				rpm -e $hbaapirpm
				HBAAPISTATUS=$?
				if [[ "$HBAAPISTATUS" = "0" ]]; then
					log_message "Uninstalling $hbaapirpm completed successfully"
				else
					log_message "An error occurred while uninstalling $hbaapirpm"
					STATUS=$FAILURE
				fi
				log_message ""
		   done
	   fi
	fi
}

uninstallElxUdevConfig()
{
	rpm -q --quiet elx-udev-config
	if [[ $? -eq 0 ]]; then
		UDEVQUERY=`rpm -q elx-udev-config`
		log_message "Uninstalling elx-udev-config: $UDEVQUERY"
		rpm -e $UDEVQUERY
		UDEVSTATUS=$?
		if [[ "$UDEVSTATUS" = "0" ]]; then
			log_message "Uninstalling $UDEVQUERY completed successfully"
		else
			log_message "An error occurred while uninstalling $UDEVQUERY"
			STATUS=$FAILURE
		fi
		log_message ""
	fi
}

printSummary()
{
	if [[ -n "$KITNAME" ]]; then
		if [[ $STATUS -eq $FAILURE ]]; then
			log_message "An error occurred while uninstalling the $PRODUCT_STRING $KITNAME Kit"
		else
			log_message "$PRODUCT_STRING $KITNAME Kit un-installation completed successfully."
			if [[ -d /${INSTALL_DIR}/scripts/ ]]; then
				rm -rf /${INSTALL_DIR}/scripts/
			fi
		fi
	else
		log_message "$PRODUCT_STRING is not installed."
	fi

	log_message ""
}

# Determine FHS support.
fhsSupport

# Initialize the log file.
initLog

# Stop EHM before uninstalling the RPMs.
stop_ocmanager

# Uninstall EHM RPMs
uninstallOcVisionSensor
uninstallElxGuiKit
uninstallElxCoreKit
uninstallElxGUI
uninstallElxJVM
uninstallElxOcmCore
uninstallElxOcmCorelibs
uninstallEnablementKit
uninstallHbaapiWrapper
uninstallElxUdevConfig
printSummary

exit $SUCCESS
