#!/bin/bash

###########################################################################
#
# lpfc_clean_xlane_conf.sh
#
# This script provided by Emulex is used to remove stale xlane luns from oas.conf
# located under /usr/sbin/lpfc/.
#
# If a xlane rule in oas.conf has port wwpn which does not match with any 
# of the port names under "/sys/class/fc_host/*/port_name", the line gets deleted.
#
# Usage:
#        lpfc_clean_xlane_conf.sh
#
###########################################################################


declare -a port_wwpns

XLANE_CONFIG_FILE="oas.conf"
XLANE_CONFIG_DIR="/usr/sbin/lpfc"

#
# Support Functions
#

collect_port_wwpn()
{
   fc_host_dir="/sys/class/fc_host"

   if [ ! "$(ls -A $fc_host_dir)" ]
   then
           echo "fc host directory empty."
                return 0
   fi

   let port_index=0

   for FC_HOST in ${fc_host_dir}/*
   do
           port_name_file=${FC_HOST}/port_name
           read port_wwpns[${port_index}] < ${port_name_file}
           port_index=${port_index}+1
   done
}

is_valid_port_wwpn()
{
    for wwpn in "${port_wwpns[@]}"
    do
        if [ "${wwpn}" == "$1" ]
        then
	   return 1
	fi
    done
    
    return 0
}

#
# Check if oas.conf file exists
#

if [ ! -e ${XLANE_CONFIG_DIR}/${XLANE_CONFIG_FILE} ]
then
    echo "${XLANE_CONFIG_DIR}/${XLANE_CONFIG_FILE} file does not exist"
fi

#
# Temporary file to copy the valid entries
#

> ${XLANE_CONFIG_DIR}/oas.conf.tmp

#
# Collect valid port wwpns
#
collect_port_wwpn

#
# Process the file
#

while read LINE
do
    wwpn=`echo ${LINE} | cut -f2 -d ' '`
    is_valid_port_wwpn ${wwpn}
    status=$?
    if [ ${status} -eq 1 ]
    then
        echo ${LINE} >> ${XLANE_CONFIG_DIR}/oas.conf.tmp
    fi
done < ${XLANE_CONFIG_DIR}/${XLANE_CONFIG_FILE}


#
# Now remove the old file and rename the temporary file
#

rm ${XLANE_CONFIG_DIR}/${XLANE_CONFIG_FILE}
mv ${XLANE_CONFIG_DIR}/oas.conf.tmp ${XLANE_CONFIG_DIR}/${XLANE_CONFIG_FILE}

exit 0
