#!/bin/sh
#/*******************************************************************
# * This file is part of the Emulex Linux Device Driver for         *
# * Fibre Channel Host Bus Adapters.                                *
# * Copyright (C) 2018 Broadcom. All Rights Reserved.  The term     *
# * "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.     *
# * Copyright (C) 2015 Emulex. All rights reserved.                 *
# *                                                                 *
# * This program is free software; you can redistribute it and/or   *
# * modify it under the terms of version 2 of the GNU General       *
# * Public License as published by the Free Software Foundation.    *
# * This program is distributed in the hope that it will be useful. *
# * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND          *
# * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,  *
# * FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT, ARE      *
# * DISCLAIMED, EXCEPT TO THE EXTENT THAT SUCH DISCLAIMERS ARE HELD *
# * TO BE LEGALLY INVALID.  See the GNU General Public License for  *
# * more details, a copy of which can be found in the file COPYING  *
# * included with this package.                                     *
# *******************************************************************/
#
# SmartSAN <command> [--ramdisk]
# Valid commands are --enable --disable --status --state --version
# The --ramdisk option, if specified, ONLY applies to --enable and --disable,
# otherwise its ignored. The script will exit with exit code 4 if the
# ramdisk generation failed.
#
# The script will exit with exit code 1 for invalid arguments
#
# This utility will error, with the following message, if the current
# running driver does not support SmartSAN.
#    SmartSAN not supported in driver
# The script will then exit with exit code 2
#
# The script will exit with exit code 3 if its running on an
# unsupported LINUX distribution. RHEL 6 and 7 as well as
# SLES 11 ans 12 are supported.
#
# -e | --enable)
# This command will enable the SmartSan module parameter.
# It will display the current state of that parameter in the running driver
# and the configured status. For example:
#    SmartSAN configured: Enabled
#
# If we try to enable and the module parameter is already enabled,
# the utility will display:
#    SmartSAN already configured: Enabled
#
# In addition the current state of the running driver will be displayed
#   SmartSAN current state: Enabled
# If not equal to the configured state, after a system reboot, or driver reload,
# the SmartSAN current state should equal SmartSAN configured state.
#
# If --ramdisk option is specified, a new ramdisk image will be automatically
# generated.
#
#
# -d | --disable)
# This command will disable the SmartSan module parameter.
# It will display the current state of that parameter in the running driver
# and the configured status. For example:
#    SmartSAN configured: Disabled
#
# If we try to disable and the module parameter is already disabled,
# the utility will display:
#    SmartSAN already configured: Disabled
#
# In addition the current state of the running driver will be displayed
#    SmartSAN current state: Disabled
# If not equal to the configured state, after a system reboot, or driver reload,
# the SmartSAN current state should equal SmartSAN configured state.
#
# If --ramdisk option is specified, a new ramdisk image will be automatically
# generated.
#
#
# -s | --status)
# This command will display the value of the SmartSAN module parameter
# as the configured status. For example:
#    SmartSAN configured: Enabled
#
#    SmartSAN configured: Disabled
#
#    SmartSAN not configured
#
#
# -t | --state)
# This command will display the current state of that parameter
# in the running driver. For example:
#    SmartSAN current state: Enabled
#
#    SmartSAN current state: Disabled
#
#
# -v | --version)
# This command will display the version of this utility
#
#

UTILITY_VERSION="Emulex SmartSAN Utility Version 1.0"

# Process --enable command
function do_enable()
{
	mfile=/etc/modprobe.d/elx-lpfc.conf
	if [[ -e "$mfile" ]]
	then
		modp=`grep lpfc_enable_SmartSAN $mfile`
		if [[ -z  "$modp" ]]
		then
			echo "options lpfc lpfc_enable_SmartSAN=1" >> $mfile
			echo SmartSAN  configured: Enabled
			echo SmartSAN current state: $CurrentString
			ConfigSAN=1
			return 0
		fi
		ConfigSAN=`echo "$modp" | sed "s/.*lpfc_enable_SmartSAN=//" | awk  '{ print $1 }'`
		if [[ ConfigSAN -eq 1 ]]
		then
			echo SmartSAN already configured: Enabled
			echo SmartSAN current state: $CurrentString
			return 1
		fi
		cat $mfile | sed "s/lpfc_enable_SmartSAN=0/lpfc_enable_SmartSAN=1/" > /tmp/SmartSAN$$
		mv /tmp/SmartSAN$$ $mfile
		chmod 664 $mfile
		echo SmartSAN  configured: Enabled
		echo SmartSAN current state: $CurrentString
	else
		echo "options lpfc lpfc_enable_SmartSAN=1" > $mfile
		echo SmartSAN  configured: Enabled
		echo SmartSAN current state: $CurrentString
	fi
	ConfigSAN=1
	return 0
}

# Process --disable command
function do_disable()
{
	mfile=/etc/modprobe.d/elx-lpfc.conf
	if [[ -e "$mfile" ]]
	then
		modp=`grep lpfc_enable_SmartSAN $mfile`
		if [[ -z  "$modp" ]]
		then
			echo "options lpfc lpfc_enable_SmartSAN=0" >> $mfile
			echo SmartSAN  configured: Disabled
			echo SmartSAN current state: $CurrentString
			ConfigSAN=1
			return 0
		fi
		ConfigSAN=`echo "$modp" | sed "s/.*lpfc_enable_SmartSAN=//" | awk  '{ print $1 }'`
		if [[ ConfigSAN -eq 0 ]]
		then
			echo SmartSAN already configured: Disabled
			echo SmartSAN current state: $CurrentString
			return 1
		fi
		cat $mfile | sed "s/lpfc_enable_SmartSAN=1/lpfc_enable_SmartSAN=0/" > /tmp/SmartSAN$$
		mv /tmp/SmartSAN$$ $mfile
		chmod 664 $mfile
		echo SmartSAN  configured: Disabled
		echo SmartSAN current state: $CurrentString
	else
		echo "options lpfc lpfc_enable_SmartSAN=0" > $mfile
		echo SmartSAN  configured: Disabled
		echo SmartSAN current state: $CurrentString
	fi
	ConfigSAN=1
	return 0
}

# Process --status command
function do_status()
{
	mfile=/etc/modprobe.d/elx-lpfc.conf
	if [[ -e "$mfile" ]]
	then
		modp=`grep lpfc_enable_SmartSAN $mfile`
		if [[ -z  "$modp" ]]
		then
			echo SmartSAN not configured
			return 0
		fi
		ConfigSAN=`echo "$modp" | sed "s/.*lpfc_enable_SmartSAN=//" | awk  '{ print $1 }'`
		if [[ ConfigSAN -eq 0 ]]
		then
			echo SmartSAN configured: Disabled
		else
			echo SmartSAN configured: Enabled
		fi
	else
		echo SmartSAN not configured
	fi
	return 0
}

# Process --state command
function do_state()
{
	get_state
	ConfigSAN=0
	echo SmartSAN current state: $CurrentString
	return 0
}

# Execute whatever is in $1.
# If $2 is present and is equal to background then run it as a background job.
# If bg'd then harvest the pid and loop on failure to send a null signal.
# Provide a dot every 3 seconds to let them know we're waiting for completion.
# After the kill fails, call wait which returns the exit status.
# If we did not bg then we're just waiting for the command to finish.
function exec_command()
{
	local pid
	if [ "$2" = background ] ; then
		eval "$1" &
		pid=$!
		while kill -0 $pid 2> /dev/null
		do
			sleep 3
			echo -n "."
		done
		echo
		wait $pid
	else
		eval "$1"
	fi
	return $?
}

# Function:  backup_files_to_elx()
#
# Description:
#   Copy each file passed as a parameter to filename.elx.
#
# Parameters:
#   A list of filenames to back up.
#
# Returns:
#   Nothing.
backup_files_to_elx()
{
	for i in $@ ; do
		if [ -e $i ] ; then
			cp -f $i $i.elx
			if [ $? -ne 0 ] ; then
				echo "Error $? copying $i to $i.elx"
			else
				echo "Original ramdisk image $i saved as $i.elx"
			fi
		fi
	done
	return 0
}

# Function:  create_ramdisk()
# Figures out what LINUX distribution we are on and Architecture
function get_distribution ()
{
    ARCH=`uname -m | sed "s/i686/i386/"`
    KERNEL=`rpm -qf /lib/modules/$(uname -r) --qf "%{VERSION}_%{RELEASE}\n"`
    UNAME=`uname -r`

    if [ -f /etc/redhat-release ] ; then
        # this is a Red Hat or XenServer Install
        if [ -f /etc/xensource-inventory ]; then
            DISTRIBUTION=xenserver
        else
            DISTRIBUTION=redhat
        fi
        RH_VERSION=$(rpm -qf /etc/redhat-release --qf "%{VERSION}" | sed "s/[^0-9\.]//g")
        RELEASE=rhel${RH_VERSION}_${KERNEL}
	BOOT_FILE=/etc/rc.d/rc.local
	MESSAGESFILE="/var/log/messages"
	if [ ${ARCH} = "ia64" ] ; then
	    KERNEL_IMAGE_PATH="/boot/efi/efi/redhat/vmlinuz-${UNAME}"
	else
	    KERNEL_IMAGE_PATH="/boot/vmlinuz-${UNAME}"
	fi
	return 0
    elif [[ -f /etc/SuSE-release || -f /etc/SUSE-brand ]]; then
        # this is a Suse install
	RELEASE_FILE_NAME="/etc/SuSE-release"
	if [ -f /etc/SUSE-brand ]; then
	    RELEASE_FILE_NAME="/etc/SUSE-brand"
	fi
	DISTRIBUTION=suse
        SUSE_VERSION=$(rpm -qf ${RELEASE_FILE_NAME} --qf "%{VERSION}" | sed "s/[^0-9\.]//g")
        RELEASE=sles${SUSE_VERSION}_${KERNEL}
	BOOT_FILE=/etc/rc.d/boot.local
	MESSAGESFILE="/var/log/messages /var/log/warn"
	if [ ${ARCH} = "ppc64" ] ; then
	    SLES_KERNEL_IMAGE=vmlinux
	    KERNEL_IMAGE_PATH="/boot/vmlinux"
	else
	    SLES_KERNEL_IMAGE=vmlinuz
	    KERNEL_IMAGE_PATH="/boot/vmlinuz-${UNAME}"
	fi
	return 0
    fi
# Unsupported distribution
    return 1
}


# Creates a ramdisk image for the currently running kernel.
function create_ramdisk()
{
    if [ ${DISTRIBUTION} = "redhat" ] ; then
	# Configure ramdisk for Red Hat
	if [ ${ARCH} = "ia64" ] ; then
	    INITRDFILE="/boot/efi/efi/${DISTRIBUTION}/initrd-${UNAME}.img"
	else
	    INITRDFILE="/boot/initramfs-${UNAME}.img"
	fi
	backup_files_to_elx ${INITRDFILE}
	echo -n "Creating ramdisk ."
	exec_command "/sbin/dracut -f ${INITRDFILE} ${UNAME}" background
	if [ $? -ne 0 ] ; then
	    echo "Could not create ramdisk image.  Restoring original ramdisk ..."
	    cp -f ${INITRDFILE}.elx ${INITRDFILE}
	    echo "Original ramdisk restored."
	    echo ""
	    echo "Please run '${INSTALL_SCRIPT} --uninstall' to remove any partially"
	    echo "installed components of the Emulex driver and utility applications."
	    exit 4
	fi
    else
	# Configure ramdisk configuration file for SuSE Linux Enterprise Server.
	if [ ${ARCH} = "ppc64" ] ; then
	    INITRDFILE="/boot/initrd"
	else
	    INITRDFILE="/boot/initrd /boot/initrd.shipped"
	fi
	backup_files_to_elx ${INITRDFILE}

	echo -n "Creating ramdisk ."
	exec_command "/sbin/mkinitrd -i initrd-${UNAME} -k ${SLES_KERNEL_IMAGE}-${UNAME} >/dev/null 2>&1" background
	if [ $? -ne 0 ] ; then
	    echo "Could not create ramdisk image.  Restoring original ramdisk ..."
	    cp -f /boot/initrd.shipped.elx /boot/initrd.shipped && cp -f /boot/initrd.elx /boot/initrd
	    if [ $? -ne 0 ] ; then
		echo "Original ramdisks restored."
	    else
		echo "Original ramdisks could not be restored - error $?"
	    fi
	    echo ""
	    echo "Please run '${INSTALL_SCRIPT} --uninstall' to remove any partially"
	    echo "installed components of the Emulex driver and utility applications."
	    exit 4
	fi
    fi
    depmod

    return 0
}

function get_state()
{
# Ensure currently running driver supports SmartSAN
	shopt -s nullglob
	lpfc_logs=/sys/class/scsi_host/*/lpfc_enable_SmartSAN
	lpfc_logs=$(echo $lpfc_logs)
	shopt -u nullglob
	if [[ -z  "$lpfc_logs" ]]
	then
		echo "SmartSAN not supported in driver"
		exit 2
	fi
	CurrentSAN=`cat $lpfc_logs | tail -1`
	if [[ $CurrentSAN -eq 1 ]]
	then
		CurrentString=Enabled
	else
		CurrentString=Disabled
	fi
	return 0
}

function usage()
{
    echo "Usage: SmartSAN --[help|enable|disable|status|state|version] [--ramdisk]"
}

# *** START of main shell script ****

# First check number of arguments
if [ $# != 1 ]
then
	if [ $# != 2 ]
	then
		usage
		exit 1
	else
		if [ "$2" != "--ramdisk" ]
		then
			usage
			exit 1
		fi
	fi
fi

# Next make sure we are on a supported distribution
get_distribution
if [[ $? != 0 ]]
then
        echo "Unable to determine distribution type"
	exit 3
fi

# Process command
ConfigSAN=0
case $1 in
-h | --help)
	usage
	exit 0
	;;
-e | --enable)
	get_state
	do_enable
	;;
-d | --disable)
	get_state
	do_disable
	;;
-s | --status)
	do_status
	exit 0
	;;
-t | --state)
	do_state
	exit 0
	;;
-v | --version)
	echo "$UTILITY_VERSION"
	exit 0
	;;
--test_ramdisk)
# undocumented test option used to test ramdisk generation
	;;
*)
	echo "ERROR: unknown parameter \"$1\""
	usage
	exit 1
	;;
esac

if [ $# == 2 ]
then
# We already verified the second argument for --ramdisk
	ConfigSAN=1
else
# Display a warning if ramdisk if different then curent state
	if [[ $ConfigSAN -eq 1 ]]
	then
		echo "Configured state of SmartSAN is different than Current state"
	fi
	ConfigSAN=0
fi

if [[ $ConfigSAN -eq 1 ]]
then
	echo Regenerating ramdisk image ...
	create_ramdisk
fi
exit 0
