#!/bin/bash

# Copyright (c) 2003-2021 Broadcom. All Rights Reserved.
# The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.

# Get environment data
OS=`uname -s`
Arch=`uname -m`
DIR="$(cd "${0%/*}" ; pwd )"
ARG_COUNT=$#

SUCCESS=0
FAILURE=1
KITNAME="Core"
PRODUCT_STRING="Emulex HBA Manager"

# Directory paths
INSTALL_DIR="usr/sbin/ocmanager"
CONF_DIR="etc"
DUMP_DIR="var/opt/emulex/ocmanager/Dump"
LOG_DIR="var/opt/emulex/ocmanager/logs"

# EHM scripts
START_EHM_SH="/${INSTALL_DIR}/start_ocmanager"
STOP_EHM_SH="/${INSTALL_DIR}/stop_ocmanager"
OP_MODE_SH="/${INSTALL_DIR}/set_operating_mode"
CONVERT_CONF_SH="/${INSTALL_DIR}/convert_conf_file"

# EHM CLI
CLI_BIN="/${INSTALL_DIR}/hbacmd"
CHKADDR_BIN="/${INSTALL_DIR}/elxchkaddr"

# EHM GUI
GUI_BIN="/${INSTALL_DIR}/ocmanager"

# Installation log
LOGFILE_NAME="utils-install.log"
LOGFILE="/${INSTALL_DIR}/${LOGFILE_NAME}"

# Configuration files
DiscConfig="/${CONF_DIR}/emulexDiscConfig"
RMConfig="/${CONF_DIR}/emulexRMConfig"
RMOptions="/${CONF_DIR}/emulexRMOptions"
RMPref="/${CONF_DIR}/emulexRMPref"
RMConf="/${CONF_DIR}/emulex/ocmanager/rm.conf"

PRINT_TO_CONSOLE=1
FHS_SUPPORT=0
DAEMONS_ENABLED="y"

# Variables & definitions used for silent installation
SILENT_ENABLED=0
SILENT_MMODE=2
SILENT_ACHANGE=1
SILENT_MHOST=0
SILENT_MEXCL=0
SILENT_MPORT=23333
SILENT_SECURITY=0
SILENT_READONLY=0
SILENT_Q0_MODE=0

# General local variables
VERIFY_MGMT_HOST=0
SECURE_PAM_MGMT_SUPPORTED=0

print()
{
	# If PRINT_TO_CONSOLE is 1, print argument #1 to the console and to the log
	# file with no trailing new line.  Otherwise, only print to the log file.
	if [[ "$PRINT_TO_CONSOLE" -eq 1 ]]; then
		printf "$1" | tee -a $LOGFILE
	else
		printf "$1" >> $LOGFILE
	fi
}

println()
{
	# If PRINT_TO_CONSOLE is 1, print argument #1 to the console and to the log
	# file with a trailing new line.  Otherwise, only print to the log file.
	if [[ "$PRINT_TO_CONSOLE" -eq 1 ]]; then
		echo "$1" | tee -a $LOGFILE
	else
		echo "$1" >> $LOGFILE
	fi
}

run()
{
	# Run the command string specified by argument #1.  If PRINT_TO_CONSOLE is 1, 
	# print stdout and stderr to both the console and the log file.  Otherwise,
	# print only to the log file.
	if [[ "$PRINT_TO_CONSOLE" -eq 1 ]]; then
		$1 2>&1 | tee -a ${LOGFILE} 2>&1
	else
		$1 >> $LOGFILE 2>&1
	fi
}

backup_hbanyware()
{
	if [[ -L /usr/sbin/hbanyware ]]; then	           # upgrade from OCM 5.1
		rm -rf /usr/sbin/hbanyware
	else
		if [[ -d /usr/sbin/hbanyware ]]; then        # upgrade from OCM 5.0
			println "Saving /usr/sbin/hbanyware as /usr/sbin/hbanyware.isave"
			rm -rf /usr/sbin/hbanyware.isave
			mv -f /usr/sbin/hbanyware /usr/sbin/hbanyware.isave
		fi
	fi
}

print_help()
{
	echo "Usage: ./install.sh [MODE [OPTION...]]"
	echo "Examples: ./install.sh -q -m=3 -a=1 -h=192.168.1.100 -e=0 -p=30000"
	echo "          ./install.sh -q -s=1"
	echo "Install or update the $PRODUCT_STRING Core Kit."
	echo ""
	echo "MODE controls the installation mode: interactive, non-interactive, or silent"
	echo "non-interactive. MODE options are:"
	echo ""
	echo "  -q0, --quiet0              Normal, fully interactive installation (default)."
	echo "                             Same as running the script with no arguments."
	echo "  -q, -q1, --quiet, --quiet1 Non-interactive installation. Users are not"
	echo "                             prompted for input during installation. Users may"
	echo "                             specify install options using the OPTION arguments."
	echo "                             Default settings will be used for all values not"
	echo "                             specified."
	echo "  -q2, --quiet2              Silent, non-interactive installation. Same as non-"
	echo "                             interactive installation, but no output will be"
	echo "                             printed to the console."   
	echo ""
	echo "OPTION arguments are used to specify the $PRODUCT_STRING installation"
	echo "settings in non-interactive and silent non-interactive mode. Default settings"
	echo "will be used for any that are not specified. The available options are:"
	echo ""
	echo "  -s, --sec=<0|1>        Secure Management (0=No, 1=Yes). Default=No."
	echo "                         When this option is selected, all EHM local and remote"
	echo "                         commands require proper EHM group membership on the" 
	echo "                         LDAP domain controller or local machine. This option"
	echo "                         overrides (and therefore is not compatible with)"
	echo "                         ALL options listed below."
	echo "  -m, --mmode=<mode>     Management mode. Numbered <mode> options are as follows:"
	echo ""
	echo  "   1   Strictly Local Management : Only manage the adapters on this host."
	echo  "                                   Management of adapters on this host from other"
	echo  "                                   hosts is not allowed."
	echo  "   2   Local Management Plus     : Only manage the adapters on this host."
	echo  "                                   Management of adapters on this host from other"
	echo  "                                   hosts is allowed. (Default)"
	echo  "   3   Full Management           : Manage the adapters on this host and other"
	echo  "                                   hosts that allow it.  Management of the adapters"
	echo  "                                   on this host from another host is allowed."
	echo  "   4   Management Host           : Manage the adapters on this host and other hosts"
	echo  "                                   that allow it.  Management of the adapters on"
	echo  "                                   this host from another host is not allowed."
	echo  "   5   Strictly Local Management : Only manage the adapters on this host."
	echo  "       with Daemons Disabled       Management of adapters on this host from other"
	echo  "                                   hosts is not allowed.  $PRODUCT_STRING"
	echo  "                                   daemons are disabled, which limits some"
	echo  "                                   application features."
	echo
	echo "  -r, --readonly=<0|1>   When set to '1', limits client users to passive (monitoring) operations;"
	echo "                         active operations (e.g. download) are prohibited. Default=0."
	echo "  -a, --achange=<0|1>    Allow users to change the management mode after"
	echo "                         installation (0=No, 1=Yes ). Default=1."
	echo "  -h, --mhost=<host>     Register with Management Host <host> (not valid for"
	echo "                         Management Modes 1 and 4). <host> may be the IPv4 or"
	echo "                         IPv6 address or hostname of a remote Management"
	echo "                         Host. '0' (default) disables this functionality."
	echo "  -e, --mexcl=<0|1>      Process requests exclusively from the Management"
	echo "                         Host (0=No, 1=Yes). Default=0."
	echo "  -p, --mport=<portnum>  TCP/IP port number of which management will occur."
	echo "                         Valid port numbers are 1024-65535. '0' will disable"
	echo "                         TCP/IP management to/from remote hosts. Default=23333"
	echo ""
}

add_group_privileges_linux()
{
	# Some files may or may not exist yet. Create them so we can set the privileges.
	# This file should not be here!
	touch $RMPref

	setfacl -m g:ocmadmin:rwx      /${INSTALL_DIR}
	setfacl -m g:ocmlocaladmin:rwx /${INSTALL_DIR}
	setfacl -m g:ocmuser:rwx       /${INSTALL_DIR}
	setfacl -m g:ocmlocaluser:rwx  /${INSTALL_DIR}

	setfacl -m g:ocmadmin:rx        $CLI_BIN
	setfacl -m g:ocmlocaladmin:rx   $CLI_BIN
	setfacl -m g:ocmuser:rx         $CLI_BIN
	setfacl -m g:ocmlocaluser:rx    $CLI_BIN

	setfacl -m g:ocmadmin:rx        $CONVERT_CONF_SH
	setfacl -m g:ocmlocaladmin:rx   $CONVERT_CONF_SH 
	setfacl -m g:ocmuser:rx         $CONVERT_CONF_SH
	setfacl -m g:ocmlocaluser:rx    $CONVERT_CONF_SH

	setfacl -m g:ocmadmin:rwx       /${INSTALL_DIR}/RMRepository
	setfacl -m g:ocmlocaladmin:rwx  /${INSTALL_DIR}/RMRepository
	setfacl -m g:ocmuser:rwx        /${INSTALL_DIR}/RMRepository
	setfacl -m g:ocmlocaluser:rwx   /${INSTALL_DIR}/RMRepository

	# [Bug 151562] - Update OCM to comply with FHS
	setfacl -m g:ocmadmin:rw        $RMConf
	setfacl -m g:ocmlocaladmin:rw   $RMConf
	setfacl -m g:ocmuser:rw         $RMConf
	setfacl -m g:ocmlocaluser:rw    $RMConf

	setfacl -m g:ocmadmin:rw        /${LOG_DIR}/cnaboardmgmt.log
	setfacl -m g:ocmlocaladmin:rw   /${LOG_DIR}/cnaboardmgmt.log
	setfacl -m g:ocmuser:rw         /${LOG_DIR}/cnaboardmgmt.log
	setfacl -m g:ocmlocaluser:rw    /${LOG_DIR}/cnaboardmgmt.log

	setfacl -R -m g:ocmadmin:rwx      /${DUMP_DIR}
	setfacl -R -m g:ocmlocaladmin:rwx /${DUMP_DIR}
	setfacl -R -m g:ocmuser:rwx       /${DUMP_DIR}
	setfacl -R -m g:ocmlocaluser:rwx  /${DUMP_DIR}

	setfacl -m g:ocmadmin:rw        /${LOG_DIR}/featuremgmt.log
	setfacl -m g:ocmlocaladmin:rw   /${LOG_DIR}/featuremgmt.log
	setfacl -m g:ocmuser:rw         /${LOG_DIR}/featuremgmt.log
	setfacl -m g:ocmlocaluser:rw    /${LOG_DIR}/featuremgmt.log

	setfacl -m g:ocmadmin:rw        /${LOG_DIR}/rm.log
	setfacl -m g:ocmlocaladmin:rw   /${LOG_DIR}/rm.log
	setfacl -m g:ocmuser:rw         /${LOG_DIR}/rm.log
	setfacl -m g:ocmlocaluser:rw    /${LOG_DIR}/rm.log

	setfacl -m g:ocmadmin:rw        /${LOG_DIR}/wwnmgmt.log
	setfacl -m g:ocmlocaladmin:rw   /${LOG_DIR}/wwnmgmt.log
	setfacl -m g:ocmuser:rw         /${LOG_DIR}/wwnmgmt.log
	setfacl -m g:ocmlocaluser:rw    /${LOG_DIR}/wwnmgmt.log

	# If misc directory doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [[ ! -d /${INSTALL_DIR}/misc ]]; then
		mkdir -p /${INSTALL_DIR}/misc
	fi
	setfacl -m g:ocmadmin:rx      /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -m g:ocmlocaladmin:rx /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -m g:ocmuser:rx       /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -m g:ocmlocaluser:rx  /${INSTALL_DIR}/misc > /dev/null 2>&1

	# bugzilla 133450
	if [[ ! -r /${INSTALL_DIR}/misc/lockFile.sem ]]; then
		touch /${INSTALL_DIR}/misc/lockFile.sem
	fi
	setfacl -m g:ocmadmin:rw      /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaladmin:rw /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -m g:ocmuser:rw       /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaluser:rw  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1

	# bugzilla 134308
	if [[ ! -r /${INSTALL_DIR}/misc/hbahost_lock.sem ]]; then
		touch /${INSTALL_DIR}/misc/hbahost_lock.sem
	fi
	setfacl -m g:ocmadmin:rw      /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaladmin:rw /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmuser:rw       /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaluser:rw  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1

	setfacl -m g:ocmadmin:rw $DiscConfig
	setfacl -m g:ocmadmin:rw $RMConfig
	setfacl -m g:ocmadmin:rw $RMOptions
	setfacl -m g:ocmadmin:rw $RMPref

	setfacl -m g:ocmlocaladmin:rw $DiscConfig
	setfacl -m g:ocmlocaladmin:rw $RMConfig
	setfacl -m g:ocmlocaladmin:rw $RMOptions
	setfacl -m g:ocmlocaladmin:rw $RMPref

	setfacl -m g:ocmuser:rw $DiscConfig
	setfacl -m g:ocmuser:rw $RMConfig
	setfacl -m g:ocmuser:rw $RMOptions
	setfacl -m g:ocmuser:rw $RMPref

	setfacl -m g:ocmlocaluser:rw $DiscConfig
	setfacl -m g:ocmlocaluser:rw $RMConfig
	setfacl -m g:ocmlocaluser:rw $RMOptions
	setfacl -m g:ocmlocaluser:rw $RMPref

	if [[ -r $GUI_BIN ]]; then
		setfacl -m g:ocmadmin:rx      $GUI_BIN
		setfacl -m g:ocmlocaladmin:rx $GUI_BIN
		setfacl -m g:ocmuser:rx       $GUI_BIN
		setfacl -m g:ocmlocaluser:rx  $GUI_BIN
	fi
	if [[ -d /${INSTALL_DIR}/jre ]]; then
		setfacl -R -m g:ocmadmin:rx      /${INSTALL_DIR}/jre
		setfacl -R -m g:ocmlocaladmin:rx /${INSTALL_DIR}/jre
		setfacl -R -m g:ocmuser:rx       /${INSTALL_DIR}/jre
		setfacl -R -m g:ocmlocaluser:rx  /${INSTALL_DIR}/jre
	fi
	if [[ -d /${INSTALL_DIR}/config ]]; then
		setfacl -m g:ocmadmin:rwx       /${INSTALL_DIR}/config
		setfacl -m g:ocmlocaladmin:rwx  /${INSTALL_DIR}/config
		setfacl -m g:ocmuser:rwx        /${INSTALL_DIR}/config
		setfacl -m g:ocmlocaluser:rwx   /${INSTALL_DIR}/config

		setfacl -m g:ocmadmin:rw        /${INSTALL_DIR}/config/*
		setfacl -m g:ocmlocaladmin:rw   /${INSTALL_DIR}/config/*
		setfacl -m g:ocmuser:rw         /${INSTALL_DIR}/config/*
		setfacl -m g:ocmlocaluser:rw    /${INSTALL_DIR}/config/*    
	fi
}

write_configuration_to_file()
{
	# write secure management' option into config file
	TF=$RMConfig.tmp
	cat $RMConfig | sed "/SecureManagement/d" > $TF
	if [[ "$SILENT_SECURITY" -eq 1 ]]; then
		echo "SecureManagement:enabled" >> $TF
	else
		echo "SecureManagement:disabled" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF

	# write server remote operation option to config file      
	TF=$RMConfig.tmp
	cat $RMConfig | sed "/RemoteOperation/d" > $TF
	if [[ "$SILENT_MMODE" -eq 1 || "$SILENT_MMODE" -eq 5 ]]; then
		echo "ServerRemoteOperation:disabled" >> $TF
	else
		echo "ServerRemoteOperation:enabled" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF

	# write client remote operation option to config file  
	TF=$DiscConfig.tmp
	cat $DiscConfig | sed "/RemoteOperation/d" > $TF
	if [[ "$SILENT_MMODE" -eq 3 || "$SILENT_MMODE" -eq 4 ]]; then
		echo "ClientRemoteOperation:enabled" >> $TF
	else
		echo "ClientRemoteOperation:disabled" >> $TF
	fi    
	cp $TF $DiscConfig
	rm $TF

	# can user make subsequent operation mode changes
	TF=$DiscConfig.tmp
	cat $DiscConfig | sed "/OperationMode/d" > $TF
	if [[ "$SILENT_ACHANGE" -eq 1 ]]; then
		echo "ClientOperationMode:enabled" >> $TF
	else
		echo "ClientOperationMode:locked" >> $TF
	fi
	cp $TF $DiscConfig
	rm $TF

	TF=$RMOptions.tmp
	cat $RMOptions | sed "/LocalHBAsOnly/d" > $TF
	if [[ "$SILENT_MMODE" -eq 3 || "$SILENT_MMODE" -eq 4 ]]; then
		echo "LocalHBAsOnly:false" >> $TF
	else
		echo "LocalHBAsOnly:true" >> $TF
	fi
	cp $TF $RMOptions
	rm $TF

	# place user's 'read only' configuration selection into config file
	TF=$DiscConfig.tmp
	cat $DiscConfig | sed "/ClientReadOnlyOperation/d" > $TF
	if [[ "$SILENT_READONLY" -eq 1 ]]; then
		echo "ClientReadOnlyOperation:enabled" >> $TF
	else
		echo "ClientReadOnlyOperation:disabled" >> $TF
	fi
	cp $TF $DiscConfig
	rm $TF

	# place 'ManagementHost' value into config file
	TF=$RMConfig.tmp
	cat $RMConfig | sed "/ManagementHost/d" > $TF
	if [[ "$SILENT_MHOST" != "0" ]]; then
		echo "ManagementHost:true" >> $TF
	else
		echo "ManagementHost:false" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF

	# place 'tcp/ip socket' port number into config file
	TF=$DiscConfig.tmp
	cat $DiscConfig | sed "/TcpSocketPortNumber/d" > $TF
	echo "TcpSocketPortNumber:$SILENT_MPORT" >> $TF
	cp $TF $DiscConfig
	rm $TF

	# place management host name or ip address value into config file
	TF=$RMConfig.tmp
	cat $RMConfig | sed "/MngmtHostIpAddress/d" > $TF
	echo "MngmtHostIpAddress:$SILENT_MHOST" >> $TF
	cp $TF $RMConfig
	rm $TF

	# place exclusive mgmnt host config value into config file
	TF=$RMConfig.tmp
	cat $RMConfig | sed "/ExclusiveHostMgmnt/d" > $TF
	if [[ "$SILENT_MEXCL" -eq 1 ]]; then
		echo "ExclusiveHostMgmnt:true" >> $TF
	else
		echo "ExclusiveHostMgmnt:false" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF

	# place 'DaemonsEnabled' boolean into config file
	TF=$RMConfig.tmp
	cat $RMConfig | sed "/DaemonsEnabled/d" > $TF
	if [[ "$DAEMONS_ENABLED" = "y" ]]; then
		echo "DaemonsEnabled:true" >> $TF
	else
		echo "DaemonsEnabled:false" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF
}

stop_ocmanager()
{
	if [[ -f ${STOP_EHM_SH} ]]; then
		echo "${STOP_EHM_SH}" >> ${LOGFILE}
		${STOP_EHM_SH} >> ${LOGFILE} 2>&1
	fi
}

oracle_version_check()
{
	oracle_major_version=`cat /etc/oracle-release | awk -F'.' '{print $1}' | awk '{print $NF}'`
	oracle_minor_version=`cat /etc/oracle-release | awk -F'.' '{print $2}' | cut -d' ' -f1`

	SECURE_PAM_MGMT_SUPPORTED=1
    
	if [[ "$oracle_major_version" -eq 7 && "$oracle_minor_version" -ge 8 ]]; then
		OS_VER="ol-7"
	elif [[ "$oracle_major_version" -eq 8 && "$oracle_minor_version" -ge 1 ]]; then
		OS_VER="ol-8"
	else
		println "This is not a supported version of ${RHEL_OS_STR}."
		println "Aborting installation."
		exit $FAILURE
	fi
}

rhel_version_check()
{
	rhel_major_version=`cat /etc/redhat-release | awk -F'.' '{print $1}' | awk '{print $NF}'`
	rhel_minor_version=`cat /etc/redhat-release | awk -F'.' '{print $2}' | cut -d' ' -f1`

	SECURE_PAM_MGMT_SUPPORTED=1
	MIN_RHEL7_VER=8
	MIN_RHEL8_VER=1
	MIN_RHEL9_VER=0
	if [[ "$Arch" = "aarch64" ]]; then
		[[ $rhel_major_version -eq 7 ]] && MIN_RHEL7_VER=6
		[[ $rhel_major_version -eq 8 ]] && MIN_RHEL8_VER=3
	fi

	if [[ "$rhel_major_version" -eq 7 && "$rhel_minor_version" -ge $MIN_RHEL7_VER ]]; then
	        OS_VER="rhel-7"
	elif [[ "$rhel_major_version" -eq 8 && "$rhel_minor_version" -ge $MIN_RHEL8_VER ]]; then
		OS_VER="rhel-8"
	elif [[ "$rhel_major_version" -eq 9 && "$rhel_minor_version" -ge $MIN_RHEL9_VER ]]; then
		OS_VER="rhel-9"
	else
		println "This is not a supported version of $RHEL_OS_STR."
		println "Aborting installation."
		exit $FAILURE
	fi
}

createLogFile()
{
	# create the installation log directory if necessary
	LOGDIRECTORY=${LOGFILE%/*}
	if [[ ! -d ${LOGDIRECTORY} ]] ; then
		mkdir -p ${LOGDIRECTORY}
	fi

	# create the installation log file if necessary
	if [[ ! -f ${LOGFILE} ]]; then
		touch ${LOGFILE}
	fi
}

platformCheck()
{
	if [[ "$OS" != "Linux" ]]; then
		println "This is not a supported platform: $OS"
		println "Aborting installation."
		exit $FAILURE
	fi
}

determineDistro()
{
	# Determine OS
	if [[ -f /etc/redhat-release ]]; then
		citrixos=0
		grep -q Citrix /etc/issue || grep -q XenServer /etc/issue
		if [[ $? -eq 0 ]]; then
			citrixos=1
		fi

		oracleos=0
		centos=`grep CentOS /etc/redhat-release | wc -l`
		scientificos=`grep Scientific /etc/redhat-release | wc -l`
		if [[ -f /etc/oracle-release ]]; then
			oracleos=`grep Oracle /etc/oracle-release | wc -l`
		fi

		ls "$DIR/$Arch" | grep -q rhel && rhel_kit=1 || rhel_kit=0

		if [[ "$oracleos" -ge 1 ]]; then
			RHEL_OS_STR="Oracle Linux"
		elif [[ "$centos" -ge 1 ]]; then
			RHEL_OS_STR="CentOS"
		elif [[ "$scientificos" -ge 1 ]]; then
			RHEL_OS_STR="Scientific Linux"
		else
			RHEL_OS_STR="Red Hat Enterprise Linux"
		fi

		uekos=`uname -a | grep uek | wc -l`
		if [[ $uekos -eq 0 && $rhel_kit -eq 0 ]] || \
		   [[ $uekos -eq 1 && $rhel_kit -eq 1 ]]; then
			println "This is not a supported version of ${RHEL_OS_STR}."
			println "Aborting installation."
			exit $FAILURE
		fi

		if [[ "$citrixos" -ge 1 ]]; then
			println "Citrix XenServer is not supported."
			println "Aborting installation."
			exit $FAILURE
		elif [[ $rhel_kit -eq 0 && $oracleos -eq 1 ]]; then
			oracle_version_check
		else
			rhel_version_check
		fi
	elif [[ -f /etc/SuSE-release ]]; then
		sles_major_version=`cat /etc/SuSE-release | grep VERSION | awk '{ print $3 }'`
		sles_minor_version=`cat /etc/SuSE-release | grep PATCHLEVEL | awk '{ print $3 }'`

		SECURE_PAM_MGMT_SUPPORTED=1

		if [[ "$sles_major_version" -eq 12 && "$sles_minor_version" -ge 5 ]]; then
			OS_VER="sles-12-sp5"
		elif [[ "$sles_major_version" -eq 12 && "$sles_minor_version" -eq 4 ]]; then
			OS_VER="sles-12-sp4"
		else
			println "This is not a supported version of SuSE Linux Enterprise Server."
			println "Aborting installation."
			exit $FAILURE
		fi
	elif [[ -f /etc/ibm_powerkvm-release ]]; then
		pkvm_major_release=`cat /etc/os-release | grep "VERSION=" | sed 's/VERSION=\"//g' | awk -F'.' '{print $1}'`
		
		if [[ "$pkvm_major_release" -eq 2 ]]; then
			# BE: use same RPMs from RHEL 7 ppc64
			OS_VER="rhel-7"
		elif [[ "$pkvm_major_release" -eq 3 ]]; then
			# LE: use same RPMs from RHEL 7 ppc64le
			OS_VER="rhel-7"
		else
			println "This is not a supported version of IBM PowerKVM."
			println "Aborting installation."
			exit $FAILURE
		fi
	elif [[ -f /etc/kylin-release ]]; then
		version_file="/etc/kylin-release"
		major_version=$(cat $version_file | awk '{ print $6 }')
		if [[ "$major_version" = "V10" ]]; then
			OS_VER="kylin-V10"
		else
			println "This is not a supported version of Kylin."
			println "Aborting installation."
			exit $FAILURE
		fi
	elif [[ -f /etc/openEuler-release ]]; then
		version_file="/etc/openEuler-release"
		major_version=$(cat $version_file | awk '{ print $3 }' | cut -d'.' -f1)
		if [[ $major_version -eq 20 ]]; then
			OS_VER="openEuler-20"
		else
			println "This is not a supported version of OpenEuler."
			println "Aborting installation."
			exit $FAILURE
		fi
	elif [[ -f /etc/os-release ]]; then
		version_file="/etc/os-release"
		sles_str=`cat $version_file | grep ^NAME= | cut -f2 -d'"'`
		if [[ -z $sles_str || "$sles_str" != "SLES" ]]; then
			println "ERROR: Cannot determine Linux distribution."
			println "Aborting installation."
			exit $FAILURE
		fi

		sles_version=`cat $version_file | grep ^VERSION_ID= | cut -f2 -d'"'`
		sles_major_version=`cat $version_file | grep ^VERSION_ID= | cut -f2 -d'"' | cut -f1 -d "."`
		if [[ $sles_version == *"."* ]]; then
			sles_minor_version=`cat $version_file | grep ^VERSION_ID= | cut -f2 -d'"' | cut -f2 -d "."`
		else
			sles_minor_version=0
		fi

		SECURE_PAM_MGMT_SUPPORTED=1

		if [[ "$sles_major_version" -eq 12 && "$sles_minor_version" -ge 5 ]]; then
			OS_VER="sles-12-sp5"
		elif [[ "$sles_major_version" -eq 12 && "$sles_minor_version" -eq 4 ]]; then
			OS_VER="sles-12-sp4"
		elif [[ "$sles_major_version" -eq 15 && "$sles_minor_version" -ge 1 ]]; then
			OS_VER="sles-15-sp1"
		else
			println "This is not a supported version of SuSE Linux Enterprise Server."
			println "Aborting installation."
			exit $FAILURE
		fi
	else
		println "ERROR: Cannot determine Linux distribution."
		println "Aborting installation."
		exit $FAILURE
	fi
}

fhsSupport()
{
	# Determine FHS support
	if [[ -f /etc/redhat-release ]]; then
		grep -q Citrix /etc/issue || grep -q XenServer /etc/issue
		if [[ $? -eq 0 ]]; then
			return
		fi

		oracleos=0
		if [[ -f /etc/oracle-release ]]; then
			oracleos=`grep Oracle /etc/oracle-release | wc -l`
		fi

		ls "$DIR/$Arch" | grep -q rhel && rhel_kit=1 || rhel_kit=0

		if [[ $rhel_kit -eq 0 && $oracleos -eq 1 ]]; then
			RELEASE_FILE="/etc/oracle-release"
		else
			RELEASE_FILE="/etc/redhat-release"
		fi

		major_version=`cat $RELEASE_FILE | awk -F'.' '{print $1}' | awk '{print $NF}'`
		minor_version=`cat $RELEASE_FILE | awk -F'.' '{print $2}' | cut -d' ' -f1`
		if [[ "$major_version" -eq 9 ]]; then
			FHS_SUPPORT=1
		fi
	fi

	if [[ $FHS_SUPPORT -eq 1 ]]; then
		INSTALL_DIR="opt/emulex/ocmanager"
		CONF_DIR="etc/emulex/ocmanager"
		BIN_DIR="${INSTALL_DIR}/bin"
		CHKADDR_BIN="/${BIN_DIR}/elxchkaddr"
		SCRIPTS_DIR="${INSTALL_DIR}/scripts"
		START_EHM_SH="/${SCRIPTS_DIR}/start_ocmanager.sh"
		STOP_EHM_SH="/${SCRIPTS_DIR}/stop_ocmanager.sh"
		OP_MODE_SH="/${SCRIPTS_DIR}/set_operating_mode.sh"
		CONVERT_CONF_SH="/${SCRIPTS_DIR}/convert_conf_file.sh"
		CLI_BIN="/${BIN_DIR}/hbacmd"
		GUI_BIN="/${BIN_DIR}/ocmanager"
		LOGFILE="/${LOG_DIR}/${LOGFILE_NAME}"

		DiscConfig="/${CONF_DIR}/emulexDiscConfig"
		RMConfig="/${CONF_DIR}/emulexRMConfig"
		RMOptions="/${CONF_DIR}/emulexRMOptions"
		RMPref="/${CONF_DIR}/emulexRMPref"
		RMConf="/${CONF_DIR}/rm.conf"
	fi
}

parseArgs()
{
	# Iterate through all command line arguments, handling them appropriately.
	for arg in "$@"
	do
		case "$arg" in
			"-q0" | "--quiet0" )
			# Normal interactive mode
			SILENT_ENABLED=0
			SILENT_Q0_MODE=1
			;;

			"-q" | "-q1" | "--quiet" | "--quiet1" )
			# Silent mode enabled
			SILENT_ENABLED=1
			;;

			"-q2" | "--quiet2" )
			# Extra silent mode enabled.  No ouput sent to console.
			SILENT_ENABLED=1
			PRINT_TO_CONSOLE=0
			;;

			"-s="* | "--sec="* )
			# Secure Management specified
			SILENT_SECURITY=`echo $arg | cut -d= -f 2`
			if [[ "$SILENT_SECURITY" -gt 1 ]]; then
				println "ERROR: Invalid secure management entry: ${SILENT_SECURITY}"
				print_help
				exit 1;
			fi
			if [[ $SECURE_PAM_MGMT_SUPPORTED -ne 1 ]]; then         
				println "ERROR: EHM Secure Management feature is not supported on this distribution."
				exit 1;
			fi
			;;

			"-m="* | "--mmode="* )
			# Management Mode specified
			SILENT_MMODE=`echo $arg | cut -d= -f 2`
			if [[ "$SILENT_MMODE" -lt 1 || "$SILENT_MMODE" -gt 5 ]]; then
				println "ERROR: Invalid mmode entry: ${SILENT_MMODE}"
				print_help
				exit 1;
			fi
			;;

			"-a="* | "--achange="* )
			# Allow Management Mode to be changed after installation
			SILENT_ACHANGE=`echo $arg | cut -d= -f 2`
			if [[ "$SILENT_ACHANGE" != "0" && "$SILENT_ACHANGE" != "1" ]]; then
				println "ERROR: Invalid achange entry: ${SILENT_ACHANGE}"
				print_help
				exit 1;
			fi
			;;

			"-h="* |"--mhost="* )
			# Management Host specified.
			SILENT_MHOST=`echo $arg | cut -d= -f 2`

			# If not "0", verify mhost entry later after the architecture-specific kits have been unpacked.
			if [[ "$SILENT_MHOST" != "0" ]]; then
				VERIFY_MGMT_HOST=1        
			fi      
			;;

			"-e="* | "--mexcl="* )
			# Specifies whether non-management hosts should process requests 
			# exclusively from the Management Host specified by mhost.
			SILENT_MEXCL=`echo $arg | cut -d= -f 2`
			if [[ "$SILENT_MEXCL" != "0" && "$SILENT_MEXCL" != "1" ]]; then
				println "ERROR: Invalid mexcl entry: ${SILENT_MEXCL}"
				print_help
				exit 1;
			fi        
			;;

			"-r="* | "--readonly="* )
			# Specifies whether ocm clients are restricted to passive (read-only)
			# operations.
			SILENT_READONLY=`echo $arg | cut -d= -f 2`
			if [[ "$SILENT_READONLY" != "0" && "$SILENT_READONLY" != "1" ]]; then
				println "ERROR: Invalid readonly entry: ${SILENT_READONLY}"
				print_help
				exit 1;
			fi        
			;;

			"-p="* | "--mport="* )
			# Set the port number by which TCP/IP management will occur, or 
			# disable TCP/IP management (mport=0).
			SILENT_MPORT=`echo $arg | cut -d= -f 2`

			# Validate the TCP/IP port number
			test_value=`echo $SILENT_MPORT | tr -d "0123456789"`
			if [[ "$test_value" != "" || "$SILENT_MPORT" -lt 1024 || "$SILENT_MPORT" -gt 65535 ]]; then
				if [[ "$SILENT_MPORT" != "0" ]]; then
					println "ERROR: Invalid mport entry: ${SILENT_MPORT}"
					print_help
					exit 1
				fi
			fi
			;;

			"-?" | "--help" )
			# Print the help message
			print_help
			exit 0
			;;

			* )
			# Default: Invalid parameter
			println "ERROR: Invalid argument: ${arg}"
			print_help
			exit 1;
			;;
		esac
	done
}

additionalArgCheck()
{
	# Return an error if the user specified additional install parameters, but didn't 
	# specify silent install (-q, -q1, -q2, --quiet, --quiet1, --quiet2). The exceptions
	# are the -q0 and --quiet0 arguments.
	if [[ "$SILENT_ENABLED" -eq 0 && $ARG_COUNT -gt 0 ]]; then
		if [[ "$#" -eq 1 && "$SILENT_Q0_MODE" -eq 1 ]]; then
			# Do nothing.
			echo ""
		else
			println "ERROR: Installation options are only accepted for non-interactive installs."
			println "       To perform a non-interactive install, add the -q, -q1, -q2, --quiet,"
			println "       --quiet1, or --quiet2 option."  
			print_help
			exit 1;
		fi
	fi
}

conflictCheck()
{
	# Check for conflicting settings (silent mode only).
	if [[ "$SILENT_ENABLED" -eq 1 ]]; then

		# Return an error if the user sets both 'secure management' mode and any other mode 
		# of operation       
		if [[ "$SILENT_SECURITY" -eq 1 ]]; then  
			if [[ "$SILENT_MMODE" -ne 2 || \
				"$SILENT_ACHANGE" -ne 1 || \
				"$SILENT_MHOST" -ne 0 || \
				"$SILENT_MEXCL" -ne 0 ||  \
				"$SILENT_READONLY" -ne 0 || \
				"$SILENT_MPORT" -ne 23333 ]]; then
				println "ERROR: When enabling 'secure management' mode, all other options "
				println "       are overridden (and therefore disallowed). "
				exit 1
			fi
		fi

		# Return an error if the user both sets a remote Management Host and sets the management 
		# mode to "Local Only" or "Management Host"       
		if [[ "$SILENT_MHOST" != "0" ]]; then  
			if [[ "$SILENT_MMODE" -eq 1 || "$SILENT_MMODE" -eq 4 ]]; then
				println "ERROR: A remote Management Host (--mhost=?) cannot be specified on a system"
				println "       configured as a Management Host (--mmode=4) or running in Local"
				println "       Only mode (--mmode=1)"
				exit 1  
			fi
		fi

		# Return an error if the user has specified exclusive management by the Management Host, yet
		# has not entered a Management Host, or is running as a Management Host, or is running in Local Only mode..
		if [[ "$SILENT_MEXCL" -eq 1 ]]; then
			if [[ "$SILENT_MMODE" -eq 4 ]]; then
				println "ERROR: Exclusive mangement by a Management Host (--mexcl=1) is not allowed on"
				println "       a Management Host (--mmode=6)."
				exit 1
			fi

			if [[ "$SILENT_MMODE" -eq 1 ]]; then
				println "ERROR: Exclusive mangement by a Management Host (--mexcl=1) cannot be specified"
				println "       in conjunction with Local Only management (--mmode=1)."
				exit 1
			fi

			if [[ "$SILENT_MHOST" = "0" ]]; then
				println "ERROR: In order to set exclusive managment by a Management Host (--mexcl=1),"
				println "       a Management Host address must be entered (--mhost=?)."
				exit 1
			fi
		fi

		# Return an error if the user configures the system as a Management Host, but also
		# requests client be read only.
		if [[ "$SILENT_READONLY" -eq 1 && "$SILENT_MMODE" -eq 4 ]]; then
			println "ERROR: read only option (--readonly=1) not valid on a Management"
			println "       Host (--mmode=4)."
			exit 1
		fi
		
		# Return an error if the user specifies both Local Only management mode and sets a TCP/IP port number
		if [[ "$SILENT_MPORT" -ne 0 && "$SILENT_MPORT" -ne 23333 && "$SILENT_MMODE" -eq 1 ]]; then
			println "ERROR: A TCP/IP port number cannot be specified (--mport=?) in conjunction"
			println "       with Local Only management (--mmode=1)."
			exit 1
		fi
	fi
}

getPkgDirectory()
{
	cd "$DIR"
	if [[ ! -d "${Arch}/${OS_VER}" ]]; then
		println "This operating system is not supported by this $PRODUCT_STRING Kit."
		println "Aborting installation."
		exit $FAILURE
	fi
}

getRpmInstallList()
{
	cd "${Arch}/${OS_VER}"
	if [[ $FHS_SUPPORT -eq 1 ]]; then
		RPMCORE=`ls elxocmcorekit-*.rpm`
		RPMGUI=`ls | grep elxocmguikit-*.rpm`
		if [[ ! -z $RPMGUI ]]; then
			KITNAME="Enterprise"
		fi
	else
		RPMCORELIBS=`ls elxocmcorelibs-*.rpm`
		RPMCORE=`ls elxocmcore-*.rpm`
		RPMGUI=`ls | grep elxocmgui-*.rpm`
		RPMJVM=`ls | grep elxocmjvm-*.rpm`
		if [[ ! -z $RPMGUI && ! -z $RPMJVM ]]; then
			KITNAME="Enterprise"
		fi
	fi
}

# Remove the OneCommandVisionSensor kit
removeOnecommandVisionSensor()
{
	if [[ "$KITNAME" == "Enterprise" ]]; then
		rpm -q --quiet OneCommandVisionSensor
		SENSOR_PRESENT=$?

		if [[ $SENSOR_PRESENT -eq 0 ]]; then
			SENSOR_RPM=`rpm -q OneCommandVisionSensor`
			if [[ $? -eq 0 ]]; then
				if [[ "$SILENT_ENABLED" -eq 1 ]]; then
					remove_sensor="y"
				else
					stay_in_loop=1
					while [[ $stay_in_loop -eq 1 ]]
					do
						println "The OneCommand Vision sensor has been detected on this system and must"
						println "be removed before $PRODUCT_STRING can be installed.  Is it okay to"
						println "remove the sensor?"
						println "Enter y to uninstall the sensor and continue installing $PRODUCT_STRING."
						println "Enter n to exit the installation."
						print   "Enter the letter 'y' or 'n': "
					
						read remove_sensor
						if [[ "$remove_sensor" = "Y" || "$remove_sensor" = "y" ]]; then
							stay_in_loop=0;
						elif [[ "$remove_sensor" = "N" || "$remove_sensor" = "n" ]]; then
							stay_in_loop=0;
						else
							println "Invalid Selection "
						fi
					done
				fi
			fi

			if [[ ! -z $SENSOR_RPM ]]; then
				if [[ "$remove_sensor" = "Y" || "$remove_sensor" = "y" ]]; then
					println "Proceeding to remove the sensor kit"
					if [[ -f /${INSTALL_DIR}/scripts/OneCommandVisionSensor*/uninstall-sensor ]]; then
						if [[ "$PRINT_TO_CONSOLE" -eq 1 ]]; then                
							/${INSTALL_DIR}/scripts/OneCommandVisionSensor*/uninstall-sensor | tee -a $LOGFILE
						else
							/${INSTALL_DIR}/scripts/OneCommandVisionSensor*/uninstall-sensor >> $LOGFILE
						fi
					else
						println "Uninstalling OneCommandVisionSensor: $SENSOR_RPM"
						if [[ "$PRINT_TO_CONSOLE" -eq 1 ]]; then 
							rpm -ev $SENSOR_RPM | tee -a $LOGFILE
						else
							rpm -ev $SENSOR_RPM >> $LOGFILE
						fi
					fi
					
					if [[ -d /${INSTALL_DIR}/scripts/OneCommandVisionSensor* ]]; then
						println "rm -rf /${INSTALL_DIR}/scripts/OneCommandVisionSensor*"
						rm -rf /${INSTALL_DIR}/scripts/OneCommandVisionSensor*
					fi
				else
					exit 1
				fi
			fi
		fi
	fi
}

coreKitOverEntKitCheck()
{
	# disallow corekit installation overtop enterprise kit (bug 107303 and 107538) 
	if [[ "$KITNAME" == "Core" ]]; then
		rpm -q --quiet elxocmgui
		if [[ $? -eq 0 ]]; then
			println "Error: Attempting to install $PRODUCT_STRING Core Kit over more fully featured $PRODUCT_STRING Enterprise Kit."
			println "If this is intentional, please uninstall Enterprise Kit before attempting Core Kit installation."
			exit -1
		fi
	fi
}

hbaapiWrapperCheck()
{
	[[ $FHS_SUPPORT -eq 1 ]] && return 0

	# Decide if libHBAAPI install is neccessary
	HBAAPI_LIST=`ls hbaapiwrapper-*`
	for hbaapi_rpm in $HBAAPI_LIST
	do
		rpm -U --test --replacepkgs $hbaapi_rpm > /dev/null 2>&1
		if [[ $? -eq 0 ]]; then
			RPMLIST="${hbaapi_rpm} ${RPMLIST}"
		fi
	done
}

ocmCoreLibsCheck()
{
	[[ $FHS_SUPPORT -eq 1 ]] && return 0

	# Decide if elxocmcorelibs install is neccessary
	TEST1=$(rpm -q --quiet HP-CNA-FC-Emulex-Enablement-Kit && echo 0 || echo 1)
	TEST2=$(rpm -q --quiet HP-CNA-FC-mezz-Emulex-Enablement-Kit && echo 0 || echo 1)
	TEST3=$(rpm -q --quiet HP-CNA-FC-sy-Emulex-Enablement-Kit && echo 0 || echo 1)
	TEST4=$(rpm -q --quiet HPE-CNA-FC-Emulex-Enablement-Kit && echo 0 || echo 1)
	TEST5=$(rpm -q --quiet HPE-CNA-FC-mezz-Emulex-Enablement-Kit && echo 0 || echo 1)
	TEST6=$(rpm -q --quiet HPE-CNA-FC-sy-Emulex-Enablement-Kit && echo 0 || echo 1)
	if [[ $TEST1 -eq 0 || $TEST2 -eq 0 || $TEST3 -eq 0 || $TEST4 -eq 0 || $TEST5 -eq 0 || $TEST6 -eq 0 ]]; then
		RPMLIST="${RPMLIST}"
	else
		RPMLIST="${RPMCORELIBS} ${RPMLIST}"
	fi
}

# Begin the EHM installation.
beginInstall()
{
	println "Beginning $PRODUCT_STRING $KITNAME Kit Installation..."
}

installEHM()
{
	# Install EHM
	RPMLIST="$RPMLIST $RPMCORE $RPMJVM $RPMGUI $RPMUDEV"
	for RPM in $RPMLIST
	do
		println "Installing ./${Arch}/${OS_VER}/$RPM"
	done
	println ""
	rpm -U --replacepkgs $RPMLIST
	STATUS=$?
	if [[ "$STATUS" != 0 ]]; then
		println "$PRODUCT_STRING $KITNAME Kit install failed."
		exit $STATUS
	fi
}

postInstallSilentCheck()
{
	#  is this a silent install
	if [[ "$SILENT_ENABLED" -eq 1 ]]; then

		# Verify the --mhost argument if entered.
		if [[ "$VERIFY_MGMT_HOST" = "1" ]]; then
			if [[ ! -f ${CHKADDR_BIN} ]]; then
				println "WARNING: Could not find elxchkaddr. Management Host address cannot"
				println "         be validated. Using anyway."
			fi

			# Call TCP/IP address verification script
			#           -1 = no address argument
			#            0 = good address
			#            1 = bad address
			#            2 = local host
			#            3 = host name not found
			${CHKADDR_BIN} $SILENT_MHOST > /dev/null 2>&1
			return_status=$?

			println ""
			println ""

			if [[ $return_status -eq 0 ]]; then
				println "Address $SILENT_MHOST successfully validated. Setting as Management Host."
			elif [[ $return_status -eq 1 ]]; then
				println "ERROR: Host address $SILENT_MHOST appears to be an invalid IP address or hostname."
				exit 1
			elif [[ $return_status -eq 2 ]]; then
				println "ERROR: Host address $SILENT_MHOST appears to be the local host address."
				println "Address should be the address of a remote machine."
				println "Please retry using a remote address or hostname."
				exit 1
			elif [[ $return_status -eq 3 ]]; then
				println ""
				println "WARNING: Management Host $SILENT_MHOST not found.  Using anyway."
				println ""
			else
				println "ERROR: Host address $SILENT_MHOST appears to be an invalid IP address or hostname."
				exit 1
			fi
		fi

		if [[ $SILENT_MMODE -eq 5 ]]; then
			DAEMONS_ENABLED="n"
		fi

		#write configuration files
		write_configuration_to_file

		if [[ $SILENT_MMODE -eq 5 ]]; then
			systemctl -q disable elxhbamgrd.service
			if [[ "$KITNAME" == "Enterprise" ]]; then
				systemctl -q disable elxdiscoveryd.service
			fi
			systemctl -q enable ehm-auto-dump.service
		else
			# start the daemons
			${START_EHM_SH} > /dev/null 2>&1
		fi
	else
		cd "${DIR}"

		# obtain user selections for remote management operating mode 
		if [[ $SECURE_PAM_MGMT_SUPPORTED -ne 1 ]]; then
			${OP_MODE_SH} 45
		else
			${OP_MODE_SH} 25
		fi

		if [[ $? -eq 1 ]]; then
			exit 1
		fi
	fi
}

setFilePermissions()
{
	#if secure management is enabled, set file permissions
	SECURE_OPERATION_ENABLED=`cat $RMConfig | grep SecureManagement:enabled`
	if [[ "$SECURE_OPERATION_ENABLED" != "" ]]; then
		# secure management file permissions changes
		add_group_privileges_linux > /dev/null 2>&1
	fi
}

removeSetOperatingModeSH()
{
	# remove script that changes operating mode if installer selects this option
	no_mode_change=`cat $DiscConfig | grep ClientOperationMode | grep locked`
	if [[ -n "$no_mode_change" ]]; then
		rm -f ${OP_MODE_SH}
	fi
}

printSummary()
{
	println "$PRODUCT_STRING $KITNAME Kit install completed successfully."

	println ""
	if [[ "$KITNAME" == "Enterprise" ]]; then
		println "Type:    ${GUI_BIN}"
		println "to run the $PRODUCT_STRING GUI client."
	else
		println "Type:    ${CLI_BIN}"
		println "to run the $PRODUCT_STRING CLI utility."
	fi
	println ""
}

fhsSupport
createLogFile
platformCheck
determineDistro
parseArgs "$@"
additionalArgCheck
conflictCheck
getPkgDirectory
getRpmInstallList
removeOnecommandVisionSensor
coreKitOverEntKitCheck
hbaapiWrapperCheck
ocmCoreLibsCheck

# Begin the EHM installation.
beginInstall

# Stop the current version of EHM before installing the RPMs.
stop_ocmanager

installEHM
postInstallSilentCheck
setFilePermissions
removeSetOperatingModeSH
backup_hbanyware
printSummary

exit $SUCCESS
  
