#!/bin/bash

# Copyright (c) 2019-2021 Broadcom. All Rights Reserved.
# The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.

SCRIPT_NAME=$(basename "$0")
SCRIPT_PATH="$( cd "$(dirname "$0")" ; pwd -P )"
OCM_KIT=$1
OCM_INSTALL_DIR=""
OCM_RPM_DIR=""
ARCH=""
PRODUCT_STRING="Emulex HBA Manager"

STANDALONE=0
REQ_ARGS=$STANDALONE

SUCCESS=0
FAILURE=1
RHEL_KIT=0
FHS_SUPPORT=0

# Set during installation
INSTALL_PATH=""
OPT_PATH=""

# Sub directories
INSTALL_PATH_SUB_DIR="ocm"
OPT_PATH_SUB_DIR="ocm_tmp"

BASE_INSTALL_DIR=""
BASE_CONF_DIR=""
BASE_OPT_DIR="var/opt/emulex/ocmanager"

TMP_DIR=$(mktemp -d /tmp/elx-hbacmd-dir.XXXXXX)

function clean_up()
{
	STATUS=$1

	rm -rf $TMP_DIR
	exit $STATUS
}

function ecd_debug()
{
	echo -e "\n[ecd_debug]"

	echo "OCM_KIT         = $OCM_KIT"
	echo "OCM_INSTALL_DIR = $OCM_INSTALL_DIR"
	echo "OCM_OPT_DIR     = $OCM_OPT_DIR"
	echo "OCM_RPM_DIR     = $OCM_RPM_DIR"
	echo "INSTALL_PATH    = $INSTALL_PATH"
	echo "OPT_PATH        = $OPT_PATH"
}

function print_help()
{
	echo "Usage:"
	if [[ $STANDALONE -eq 1 ]]; then
		echo "$SCRIPT_NAME <elxocmcore tgz file>"
	else
		echo "$SCRIPT_NAME"
	fi

	clean_up $SUCCESS
}

function corekit_check()
{
	echo -e "\n[corekit_check]"

	if [[ -d "$OCM_KIT" ]]; then
		echo "ERROR: Invalid argument: path is a directory"
		clean_up $FAILURE
	fi

	if [[ ! -f "$OCM_KIT" ]]; then
		echo "ERROR: Invalid argument: file not found"
		clean_up $FAILURE
	fi

	KIT_TGZ="$(basename $OCM_KIT)"
	echo $KIT_TGZ | grep elxocmcore | grep -q .tgz
	if [[ $? -ne 0 ]]; then
		echo "ERROR: This script supports the elxocmcore kit only"
		clean_up $FAILURE
	fi

	echo "Core Kit: $(basename $OCM_KIT)"
}

function get_install_dir()
{
	echo -e "\n[get_install_dir]"

	stay_in_loop=1
	while [[ $stay_in_loop -eq 1 ]]
	do
		echo -n "Enter the installation directory path: "
		read INSTALL_PATH
		if [[ -d "$INSTALL_PATH/$INSTALL_PATH_SUB_DIR" ]]; then
			echo "$INSTALL_PATH contains a directory called $INSTALL_PATH_SUB_DIR"
			echo "Choose another directory"
		else
			mkdir -p "$INSTALL_PATH/$INSTALL_PATH_SUB_DIR"
			if [[ $? -eq 0 ]]; then
				stay_in_loop=0
			else
				echo "Directory could not be created"
			fi
		fi
	done

	INSTALL_PATH="$INSTALL_PATH/$INSTALL_PATH_SUB_DIR"
	echo "Installation directory: $INSTALL_PATH"
}

function get_tmp_dir()
{
	echo -e "\n[get_tmp_dir]"

	stay_in_loop=1
	while [[ $stay_in_loop -eq 1 ]]
	do
		echo -n "Enter the temp directory path: "
		read OPT_PATH
		if [[ -d "$OPT_PATH/$OPT_PATH_SUB_DIR" ]]; then
			echo "$OPT_PATH contains a directory called $OPT_PATH_SUB_DIR"
			echo "Choose another directory"
		else
			mkdir -p "$OPT_PATH/$OPT_PATH_SUB_DIR"
			if [[ $? -eq 0 ]]; then
				stay_in_loop=0
			else
				echo "Directory could not be created"
			fi
		fi
	done

	OPT_PATH="$OPT_PATH/$OPT_PATH_SUB_DIR"
	echo "Temp directory: $OPT_PATH"
}

function untar_ocm()
{
	echo -e "\n[untar_ocm]"

	if [[ $STANDALONE -eq 1 ]]; then
		echo "Untarring $(basename $OCM_KIT)"
		tar -zxf $OCM_KIT -C $TMP_DIR || clean_up $FAILURE
	else
		cp -r $(pwd) ${TMP_DIR}/ || clean_up $FAILURE
	fi

	OCM_KIT_DIR=$(ls $TMP_DIR)

	OCM_KIT_DIR="$TMP_DIR/$OCM_KIT_DIR"
	echo "Output directory: $OCM_KIT_DIR"
}

function get_arch()
{
	ARCH=$(uname -m)
	case ${ARCH} in
		x86_64)
			;;

		ppc64)
			;;

		ppc64le)
			;;

		*)
			echo "ERROR: Unsupported architecture."
			echo "Aborting installation."
			clean_up $FAILURE
			;;
	esac
}

function oracle_version_check()
{
	RHEL_OS_STR="Oracle Linux"

	ol_major_version=`cat /etc/oracle-release | awk -F'.' '{print $1}' | awk '{print $NF}'`
	ol_minor_version=`cat /etc/oracle-release | awk -F'.' '{print $2}' | cut -d' ' -f1`

	if [[ $ol_major_version -eq 7 && $ol_minor_version -ge 4 ]]; then
		OS_VER="ol-7"
	elif [[ $ol_major_version -eq 8 && $ol_minor_version -ge 0 ]]; then
		OS_VER="ol-8"
	else
		echo "ERROR: This is not a supported version of $RHEL_OS_STR."
		echo "Aborting installation."
		clean_up $FAILURE
	fi
}

function rhel_version_check()
{
	centos=`grep CentOS /etc/redhat-release | wc -l`
	scientificos=`grep Scientific /etc/redhat-release | wc -l`

	if [[ "$centos" == "1" ]]; then
		RHEL_OS_STR="CentOS"
	elif [[ "$scientificos" == "1" ]]; then
		RHEL_OS_STR="Scientific Linux"
	else
		RHEL_OS_STR="Red Hat Enterprise Linux"
	fi

	rhel_major_version=`cat /etc/redhat-release | awk -F'.' '{print $1}' | awk '{print $NF}'`
	rhel_minor_version=`cat /etc/redhat-release | awk -F'.' '{print $2}' | cut -d' ' -f1`

	MIN_RHEL8_VER=0
	MIN_RHEL9_VER=0

	if [[ $rhel_major_version -eq 7 && $rhel_minor_version -ge 7 ]]; then
		OS_VER="rhel-7"
	elif [[ $rhel_major_version -eq 8 && $rhel_minor_version -ge $MIN_RHEL8_VER ]]; then
		OS_VER="rhel-8"
	elif [[ $rhel_major_version -eq 9 && $rhel_minor_version -ge $MIN_RHEL9_VER ]]; then
		OS_VER="rhel-9"
		FHS_SUPPORT=1
	else
		echo "ERROR: This is not a supported version of $RHEL_OS_STR."
		echo "Aborting installation."
		clean_up $FAILURE
	fi
}

function get_ocm_rpm_dir()
{
	echo -e "\n[get_ocm_rpm_dir]"

	if [[ -f /etc/oracle-release ]]; then
		uekos=`uname -a | grep uek | wc -l`
		ls $ARCH | grep -q rhel && RHEL_KIT=1

		if [[ $uekos -eq 0 && $RHEL_KIT -eq 0 ]] || \
		   [[ $uekos -eq 1 && $RHEL_KIT -eq 1 ]]; then
			echo "ERROR: This is not a supported version of Oracle Linux."
			echo "Aborting installation."
			clean_up $FAILURE
		fi
	fi

	cd "$OCM_KIT_DIR"
	if [[ -f /etc/oracle-release && $RHEL_KIT -eq 0 ]]; then
		oracle_version_check
	elif [[ -f /etc/redhat-release ]]; then
		grep -q Citrix /etc/issue || grep -q XenServer /etc/issue
		if [[ $? -eq 0 ]]; then
			echo "ERROR: Citrix XenServer is not supported."
			echo "Aborting installation."
			clean_up $FAILURE
		fi

		rhel_version_check
	elif [[ -f /etc/os-release ]]; then
		version_file="/etc/os-release"
		sleslevel=`grep CPE_NAME /etc/os-release | cut -d ':' -f3`

		if [[ ${sleslevel} == "suse" ]]; then
			sles_major_version=`grep CPE_NAME $version_file | cut -d'"' -f2 | cut -d':' -f5`
			sles_minor_version=`grep CPE_NAME $version_file | cut -d'"' -f2 | cut -d':' -f6`
			if [[ -z $sles_minor_version ]]; then
				sles_minor_version=0
			else
				sles_minor_version=`echo $sles_minor_version | cut -c 3-`
			fi
		else
			echo "ERROR: Cannot determine Linux distribution."
			echo "Aborting installation."
			clean_up $FAILURE
		fi

		if [[ $sles_major_version -eq 12 && $sles_minor_version -ge 5 ]]; then
			OS_VER="sles-12-sp5"
		elif [[ $sles_major_version -eq 12 && $sles_minor_version -eq 4 ]]; then
			OS_VER="sles-12-sp4"
		elif [[ $sles_major_version -eq 15 && $sles_minor_version -ge 1 ]]; then
			OS_VER="sles-15-sp1"
		else
			echo "ERROR: This is not a supported version of SuSE Linux Enterprise Server."
			echo "Aborting installation."
			clean_up $FAILURE
		fi
	else
		echo "ERROR: Cannot determine Linux distribution."
		echo "Aborting installation."
		clean_up $FAILURE
	fi

	OCM_RPM_DIR="$OCM_KIT_DIR/${ARCH}/${OS_VER}"
	if [[ ! -d $OCM_RPM_DIR ]]; then
		echo "ERROR: $OCM_RPM_DIR does not exist"
		echo "This operating system is not supported by this $PRODUCT_STRING Kit."
		echo "Aborting installation."
		clean_up $FAILURE
	fi

	echo "EHM RPM directory: $OCM_RPM_DIR"
}

function set_dir_paths()
{
	if [[ $FHS_SUPPORT -eq 1 ]]; then
		BASE_INSTALL_DIR="opt/emulex/ocmanager"
		BASE_CONF_DIR="etc/emulex/ocmanager"
	else
		BASE_INSTALL_DIR="usr/sbin/ocmanager"
		BASE_CONF_DIR="etc"
	fi
}

function extract_ocm_rpms()
{
	echo -e "\n[extract_ocm_rpms]"

	cd $OCM_RPM_DIR
	SUPPORTED_MAJOR=12
	SUPPORTED_MINOR=6

	rpms=$(ls)
	for rpm in $rpms
	do
		echo $rpm | grep -q 32bit
		if [[ $? -ne 0 ]]; then
			echo "rpm2cpio $rpm | cpio -divm"
			rpm2cpio $rpm | cpio -divm > /dev/null 2>&1
		fi

		rm -f $rpm
	done
}

function rm_ocm_symlinks
{
	echo -e "\n[rm_ocm_symlinks]"

	cd $OCM_RPM_DIR

	echo "Deleting EHM symbolic links"

	if [[ $FHS_SUPPORT -eq 1 ]]; then
		rm -f $BASE_INSTALL_DIR/Dump
		rm -f $BASE_INSTALL_DIR/logs
		rm -f $BASE_INSTALL_DIR/rm.conf
	else
		rm -f $(find usr/sbin/ -type l)
	fi
}

function rm_ocm_scripts()
{
	echo -e "\n[rm_ocm_scripts]"

	cd $OCM_RPM_DIR

	echo "Deleting EHM executables that are not needed"
	if [[ $FHS_SUPPORT -eq 1 ]]; then
		rm -f $BASE_INSTALL_DIR/bin/dfc
		rm -f $BASE_INSTALL_DIR/bin/elxchkaddr
		rm -f $BASE_INSTALL_DIR/bin/elxhbamgrd
		rm -f $BASE_INSTALL_DIR/scripts/get_udev_name.sh
		rm -f $BASE_INSTALL_DIR/scripts/set_operating_mode.sh
		rm -f $BASE_INSTALL_DIR/scripts/start_ocmanager.sh
		rm -f $BASE_INSTALL_DIR/scripts/stop_ocmanager.sh
		rm -f $BASE_INSTALL_DIR/scripts/set_operating_mode.sh
		rm -f $BASE_INSTALL_DIR/scripts/uninstall_ocmanager.sh
		rm -rf $BASE_INSTALL_DIR/RMRepository
		rm -rf etc/ld.so.conf.d/
		rm -rf usr/lib
	else
		rm -f $BASE_INSTALL_DIR/scripts/uninstall_ocmanager.sh
		rm -f $BASE_INSTALL_DIR/elxchkaddr
		rm -f $BASE_INSTALL_DIR/elxhbamgrd
		rm -f $BASE_INSTALL_DIR/set_operating_mode
		rm -f $BASE_INSTALL_DIR/start_ocmanager
		rm -f $BASE_INSTALL_DIR/stop_ocmanager
		rm -rf etc/init.d
		rm -rf etc/logrotate.d
		rm -rf usr/lib
	fi
}

function set_ocm_install_dir
{
	echo -e "\n[set_ocm_install_dir]"

	OCM_INSTALL_DIR="$INSTALL_PATH/$BASE_INSTALL_DIR"

	echo "EHM installation directory: $OCM_INSTALL_DIR"
}

function set_ocm_tmp_dir
{
	echo -e "\n[set_ocm_tmp_dir]"

	OCM_OPT_DIR="$OPT_PATH/$BASE_OPT_DIR"

	echo "EHM temporary directory: $OCM_OPT_DIR"
}

function update_emulexRMConfig()
{
	echo -e "\n[update_emulexRMConfig]"

	cd $OCM_RPM_DIR
	TMP_FILE=$(mktemp /tmp/elx-hbacmd-emulexRMConfig.XXXXXX)

	emulexRMConfig="$BASE_CONF_DIR/emulexRMConfig"
	[[ ! -f $emulexRMConfig ]] && clean_up $FAILURE

	grep InstallDir $emulexRMConfig -v \
		| grep -v OptDir \
		| grep -v RepositoryDir \
		| grep -v LogFilePath \
		| grep -v LastFWDir > $TMP_FILE

	echo "Updating $emulexRMConfig"

	echo "InstallDir:$OCM_INSTALL_DIR" >> $TMP_FILE
	echo "OptDir:$OCM_OPT_DIR" >> $TMP_FILE
	echo "RepositoryDir:$OCM_OPT_DIR/RMRepository" >> $TMP_FILE
	echo "LastFWDir:$OCM_OPT_DIR/RMRepository" >> $TMP_FILE
	echo "LogFilePath:$OCM_OPT_DIR/logs/rm.log" >> $TMP_FILE

	rm -f $emulexRMConfig
	cp $TMP_FILE $emulexRMConfig
	rm -f $TMP_FILE
}

function mk_RMRepository_dir()
{
	echo -e "\n[mk_RMRepository_dir]"

	cd $OCM_RPM_DIR

	echo "mkdir -p $BASE_OPT_DIR/RMRepository"
	mkdir -p $BASE_OPT_DIR/RMRepository
}

function mk_hbacmd_script()
{
	echo -e "\n[mk_hbacmd_script]"

	cd $SCRIPT_PATH
	TMP_FILE=$(mktemp /tmp/elx-hbacmd-sh.XXXXXX)

	echo "Creating elx_hbacmd.sh"
	sed "s@__INSTALL_PATH__@${INSTALL_PATH}@" elx_hbacmd.sh.in > $TMP_FILE
	sed "s@__OPT_PATH__@${OPT_PATH}@" $TMP_FILE > elx_hbacmd.sh
	sed "s@__INSTALL_OS__@${OS_VER}@" elx_hbacmd.sh > $TMP_FILE
	sed "s@__INSTALL_ARCH__@${ARCH}@" $TMP_FILE > elx_hbacmd.sh
	sed "s@__RHEL_KIT__@${RHEL_KIT}@" elx_hbacmd.sh > $TMP_FILE
	sed "s@__FHS_SUPPORT__@${FHS_SUPPORT}@" $TMP_FILE > elx_hbacmd.sh 
	#mv $TMP_FILE elx_hbacmd.sh
	chmod +x elx_hbacmd.sh
}

function cp_ocm_to_dest_dir()
{
	echo -e "\n[cp_ocm_to_dest_dir]"

	echo "cp -r $OCM_RPM_DIR/* $INSTALL_PATH/"
	cp -r $OCM_RPM_DIR/* $INSTALL_PATH/ || clean_up $FAILURE

	echo "mv elx_hbacmd.sh $OCM_INSTALL_DIR/"
	mv elx_hbacmd.sh $OCM_INSTALL_DIR/ || clean_up $FAILURE
}

function install_complete()
{
	echo -e "\n[install_complete]"

	echo "Installation complete"
	echo "EHM installation directory: $OCM_INSTALL_DIR"
	echo "EHM temporary directory   : $OPT_PATH/$BASE_OPT_DIR"
}

[[ $# -ne $REQ_ARGS ]] && print_help
if [[ $STANDALONE -eq 1 ]]; then
	corekit_check
fi
untar_ocm
get_arch
get_ocm_rpm_dir
set_dir_paths
extract_ocm_rpms
get_install_dir
get_tmp_dir
rm_ocm_symlinks
rm_ocm_scripts
set_ocm_install_dir
set_ocm_tmp_dir
update_emulexRMConfig
mk_RMRepository_dir
mk_hbacmd_script
cp_ocm_to_dest_dir
install_complete
#ecd_debug

clean_up $SUCCESS
