#!/usr/bin/bash
# 
# Copyright (c) 2003-2021 Broadcom. All Rights Reserved.
# The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
# 

PRODUCT_STRING="Emulex HBA Manager"
platform_os=`uname -s`
FHS_SUPPORT=1
mode_of_operation="3"
enable_tcpip_operation="y"
secure_management_mode="u"
change_mode_of_operation="y"
read_only_mode_of_operation="n"
daemons_enabled="y"
tcp_socket_change="n"
socket_port_value="23333"
management_host_address=""
management_host_mode="n"
management_host_is_exclusive="n"
user_input="y"
install_called=$1
undefined_groups=""

if [ -z $install_called ]; then
	install_called=0
fi

setDirPathsSolaris()
{
	if [ "$platform_os" != "Solaris" ]; then
		return
	fi

	INSTALL_DIR="opt/ELXocm"
	CONF_DIR="etc"
	BIN_DIR="${INSTALL_DIR}"
	LOG_DIR="${INSTALL_DIR}"
	DUMP_DIR="${INSTALL_DIR}/Dump"

	START_EHM_SH="/${INSTALL_DIR}/start_ocmanager"
	STOP_EHM_SH="/${INSTALL_DIR}/stop_ocmanager"
	CONVERT_CONF_SH="/${INSTALL_DIR}/convert_conf_file"

	CLI_BIN="/${BIN_DIR}/hbacmd"
	GUI_BIN="/${BIN_DIR}/ocmanager"
	CHKADDR_BIN="/${BIN_DIR}/elxchkaddr"

	RMConf="/${CONF_DIR}/rm.conf"
	DiscConfig="/${CONF_DIR}/emulexDiscConfig"
	RMConfig="/${CONF_DIR}/emulexRMConfig"
	RMOptions="/${CONF_DIR}/emulexRMOptions"
	RMPref="/${CONF_DIR}/emulexRMPref"
}

setDirPathsLinux()
{
	if [[ "$platform_os" != "Linux" ]]; then
		return
	fi

	VAR_DIR="var/opt/emulex"
	LOG_DIR="${VAR_DIR}/ocmanager/logs"
	DUMP_DIR="${VAR_DIR}/ocmanager/Dump"

	if [[ $FHS_SUPPORT -eq 1 ]]; then
		INSTALL_DIR="opt/emulex/ocmanager"
		CONF_DIR="etc/emulex/ocmanager"
		BIN_DIR="${INSTALL_DIR}/bin"
		SCRIPTS_DIR="${INSTALL_DIR}/scripts"

		RM_CONF_DIR="${CONF_DIR}"

		START_EHM_SH="/${SCRIPTS_DIR}/start_ocmanager.sh"
		STOP_EHM_SH="/${SCRIPTS_DIR}/stop_ocmanager.sh"
		CONVERT_CONF_SH="/${SCRIPTS_DIR}/convert_conf_file.sh"

		LOG_FILE="/${LOG_DIR}/${LOGFILE_NAME}"
	else
		INSTALL_DIR="usr/sbin/ocmanager"
		CONF_DIR="etc"
		BIN_DIR="${INSTALL_DIR}"
		SCRIPTS_DIR="${INSTALL_DIR}"

		RM_CONF_DIR="etc/emulex/ocmanager"

		START_EHM_SH="/${INSTALL_DIR}/start_ocmanager"
		STOP_EHM_SH="/${INSTALL_DIR}/stop_ocmanager"
		CONVERT_CONF_SH="/${INSTALL_DIR}/convert_conf_file"

		LOG_FILE="/${INSTALL_DIR}/${LOGFILE_NAME}"
	fi

	CLI_BIN="/${BIN_DIR}/hbacmd"
	GUI_BIN="/${BIN_DIR}/ocmanager"
	CHKADDR_BIN="/${BIN_DIR}/elxchkaddr"

	DiscConfig="/${CONF_DIR}/emulexDiscConfig"
	RMConfig="/${CONF_DIR}/emulexRMConfig"
	RMOptions="/${CONF_DIR}/emulexRMOptions"
	RMPref="/${CONF_DIR}/emulexRMPref"
	RMConf="/${RM_CONF_DIR}/rm.conf"
}

setDirPaths()
{
	if [ "$platform_os" = "Linux" ]; then
		setDirPathsLinux
	else
		setDirPathsSolaris
	fi
}

enable_elxhbamgr()
{
	if [ "$platform_os" = "Linux" ]; then
		systemctl -q enable elxhbamgrd.service
	elif [ "$platform_os" = "SunOS" ];then
		# Permanently enable the elxhbamgrd daemon so that it initializes at boot
		svcadm enable svc:/application/elxhbamgrd:default > /dev/null 2>&1   
	fi
}

enable_elxdisco()
{
	if [ "$platform_os" = "Linux" ]; then
		if [ -f /usr/lib/systemd/system/elxdiscoveryd.service ]; then
			systemctl -q enable elxdiscoveryd.service
		fi
	elif [ "$platform_os" = "SunOS" ]; then
		# Permanently enable the elxdiscoveryd daemon so that it initializes at boot
		svcadm enable svc:/application/elxdiscoveryd:default > /dev/null 2>&1
	fi
}

disable_elxhbamgr()
{
	if [ "$platform_os" = "Linux" ]; then
		systemctl -q disable elxhbamgrd.service
	elif [ "$platform_os" = "SunOS" ]; then
		# Permanently disable the elxhbamgr daemon (until re-enabled by the user)    
		svcadm disable svc:/application/elxhbamgrd:default > /dev/null 2>&1    
	fi
}

disable_elxdisco()
{
	if [ "$platform_os" = "Linux" ]; then
		if [ -f /usr/lib/systemd/system/elxdiscoveryd.service ]; then
			systemctl -q disable elxdiscoveryd.service
		fi
	elif [ "$platform_os" = "SunOS" ]; then
		# Permanently disable the elxdiscoveryd daemon (until re-enabled by the user)
		svcadm disable svc:/application/elxdiscoveryd:default > /dev/null 2>&1
	fi
}

enable_ehm_autodump()
{
	if [ "$platform_os" = "Linux" ]; then
		if [ -f /usr/lib/systemd/system/ehm-auto-dump.service ]; then
			systemctl -q enable ehm-auto-dump.service
		fi
	fi
}

disable_ehm_autodump()
{
	if [ "$platform_os" = "Linux" ]; then
		if [ -f /usr/lib/systemd/system/ehm-auto-dump.service ]; then
			systemctl -q disable ehm-auto-dump.service
		fi
	fi
}

question_enable_tcpip()
{
	stay_in_loop=1
	while [ $stay_in_loop -eq 1 ]
	do
		echo ""
		if [ "$mode_of_operation" = "2" ];then
			echo "Do you want to enable TCP/IP Management from remote hosts? (Y/N)" 
		else
			echo "Do you want to enable TCP/IP Management to/from remote hosts? (Y/N)"
		fi 
		echo ""
		echo "Enter 'y' to enable TCP/IP remote management (default)."
		echo "Enter 'n' to disable TCP/IP remote management."
		echo 
		printf "Enter the letter 'y' or 'n': "

		# Obtain user selection for Emulex HBA Manager client read only mode of operation
		read enable_tcpip_operation
		if [ "$enable_tcpip_operation" = "N" ] || [ "$enable_tcpip_operation" = "n" ]; then
			enable_tcpip_operation="n"
			socket_port_value="0"
			echo "You selected: disable TCP/IP remote management "
			stay_in_loop=0;
		elif [ "$enable_tcpip_operation" = "Y" ] || [ "$enable_tcpip_operation" = "y" ]; then
			enable_tcpip_operation="y"
			echo "You selected: enable TCP/IP remote management "
			stay_in_loop=0;
		else
			echo "Invalid Selection "
		fi
	done
}

question_tcp_port_number()
{
	stay_in_loop=1
	while [ $stay_in_loop -eq 1 ]
	do
		echo ""

		printf "Enter TCP/IP port number to use or blank for default (23333): "

		# Obtain user selection for Emulex HBA Manager tcp port number
		read socket_port_value
		if [ "$socket_port_value" = "" ];then
			stay_in_loop=0;
			socket_port_value="23333"
		fi

		# verify user did not enter any non-digit in integer string
		test_value=`echo $socket_port_value | tr -d "0123456789"`
		if [ "$test_value" == "" ];then
			if [ $((socket_port_value)) -ge 1024 ];then
				if [ $((socket_port_value)) -le 65536 ];then
					stay_in_loop=0;
				fi
			fi
		fi
		if [ $stay_in_loop -eq 1 ];then
			echo "Invalid entry; valid port number must be decimal integer value > 1024 and < 65536 "
		fi
	done
}

question_management_host_address()
{
	stay_in_loop=1
	while [ $stay_in_loop -eq 1 ]
	do
		echo ""
		printf "Enter management host address (blank for none): " 
		# Obtain user selection for Emulex HBA Manager remote IP address or host name
		read management_host_address
		if [ "$management_host_address" = "" ];then
			stay_in_loop=0;
			management_host_address=""
		else
			# call verification script
			#           -1 = no address argument
			#            0 = good address
			#            1 = bad address
			#            2 = local host
			#            3 = host name not found
			${CHKADDR_BIN} $management_host_address
			return_status=$?
			if [ $return_status -eq 0 ]; then
				stay_in_loop=0;     # good address
			else
				if [ $return_status -eq 1 ]; then
					echo "Host address $management_host_address appears to be an invalid IP address or host name."
					echo "Please correct address and re-enter.\n"
				else
					if [ $return_status -eq 2 ]; then
						echo "Host address $management_host_address appears to be the local host address."
						echo "Address should be address of remote machine."
						echo "Please re-enter using remote address or hostname."
					else
						if [ $return_status -eq 3 ]; then
							stay_in_loop2=1
							while [ $stay_in_loop2 -eq 1 ]
							do
								echo ""
								echo "Host name $management_host_address not found."
								echo "Do you want to use this host name anyway? (Y/N)"
								echo 
								printf "Enter the letter 'y' or 'n': "

								# Obtain user selection 
								read user_input
								if [ "$user_input" = "N" ] || [ "$user_input" = "n" ]; then
									stay_in_loop2=0
								elif [ "$user_input" = "Y" ] || [ "$user_input" = "y" ]; then
									echo "You selected host name $management_host_address "
									stay_in_loop2=0
									stay_in_loop=0
								else
									echo "Invalid Selection, enter either 'y' or 'n'."
								fi 
							done
						else
							echo "Host address $management_host_address appears to be an invalid IP address or host name."
							echo "Please correct address and re-enter.\n"
						fi
					fi
				fi
			fi
		fi
	done
}

question_exclusive_host()
{
	stay_in_loop=1
	while [ $stay_in_loop -eq 1 ]
	do
		echo ""
		echo ""
		echo "Exclude management of this host from any other host"
		echo "(other than management host) (y/n)?"
		echo ""
		echo "Enter 'y' to limit remote management to ONLY the management host (default)."
		echo "Enter 'n' to allow other remote hosts to manage this host"
		echo "          in addition to the management host."
		echo 
		printf "Enter the letter 'y' or 'n': "

		# Obtain user selection for Emulex HBA Manager exclusive remote oob management
		read management_host_is_exclusive
		if [ "$management_host_is_exclusive" = "N" ] || [ "$management_host_is_exclusive" = "n" ]; then
			management_host_is_exclusive="n"
			echo "You selected: no exclusive host management "
			stay_in_loop=0;
		elif [ "$management_host_is_exclusive" = "Y" ] || [ "$management_host_is_exclusive" = "y" ]; then
			management_host_is_exclusive="y"
			echo "You selected: exclusive host management "
			stay_in_loop=0;
		else
			echo "Invalid Selection "
		fi
	done
}

question_secure_management_mode()
{
	stay_in_loop=1
	while [ $stay_in_loop -eq 1 ]
	do
		echo ""
		echo "Do you want to enable Secure Management feature for $PRODUCT_STRING? (s/u)" 
		echo ""
		echo "The secure management feature requires $PRODUCT_STRING groups be configured on"
		echo "the LDAP network or the local host machine to provide for $PRODUCT_STRING operation."
		echo ""
		echo "Enter 's' to select secure management. (LDAP/NIS EHM group configuration required)"
		echo "Enter 'u' to run without secure management (default). "
		echo
		printf "Enter the letter 's' or 'u': "

		# Obtain user selection for Emulex HBA Manager client read only mode of operation
		read secure_management_mode
		if [ "$secure_management_mode" = "U" ] || [ "$secure_management_mode" = "u" ]; then
			secure_management_mode="u"
			echo "You selected: Secure Management Disabled"
			stay_in_loop=0;
		elif [ "$secure_management_mode" = "S" ] || [ "$secure_management_mode" = "s" ]; then
			secure_management_mode="s"
			echo "You selected: Secure Management Enabled"
			stay_in_loop=0;
		else
			echo "Invalid Selection "
		fi
	done
}

question_management_mode()
{
	stay_in_loop=1
	while [ $stay_in_loop -eq 1 ]
	do     
		echo
		echo "Select desired mode of operation for $PRODUCT_STRING"
		echo
		echo "   1   Strictly Local Management : Only manage the adapters on this host."
		echo "                                   Management of adapters on this host from other"
		echo "                                   hosts is not allowed."
		echo "   2   Local Management Plus     : Only manage the adapters on this host."
		echo "                                   Management of adapters on this host from other"
		echo "                                   hosts is allowed."
		echo "   3   Full Management           : Manage the adapters on this host and other"
		echo "                                   hosts that allow it.  Management of the adapters"
		echo "                                   on this host from another host is allowed."
		echo "   4   Management Host           : Manage the adapters on this host and other hosts"
		echo "                                   that allow it.  Management of the adapters on"
		echo "                                   this host from another host is not allowed."
if [ "$platform_os" = "Linux" ];then
		echo "   5   Strictly Local Management : Only manage the adapters on this host."
		echo "       with Daemons Disabled       Management of adapters on this host from other"
		echo "                                   hosts is not allowed.  $PRODUCT_STRING"
		echo "                                   daemons are disabled, which limits some"
		echo "                                   application features."
		echo
		printf "Enter the number 1, 2, 3, 4, or 5: "
else
		printf "Enter the number 1, 2, 3, or 4: "
fi

		# Obtain user selection for Emulex HBA Manager mode of operation
		read mode_of_operation
		if [ "$mode_of_operation" = "1" ];then
			echo "You selected: 'Local Only Mode' "

			# On Linux:
			# EHM daemons are controlled by systemd.
			# By default, elxdiscoveryd does not start at time.

			# Turn off discovery daemon
			disable_elxdisco

			# Turn off auto dump
			disable_ehm_autodump

			# Turn on rmserver daemon
			enable_elxhbamgr
			stay_in_loop=0

		elif [ "$mode_of_operation" = "2" ];then
			echo "You selected: 'Managed-only Mode' "

			question_enable_tcpip

			if [ "$enable_tcpip_operation" = "y" ]; then
				question_management_host_address
				if [ "$management_host_address" != "" ];then
					question_exclusive_host    
				fi
				question_tcp_port_number
			fi

			# On Linux:
			# EHM daemons are controlled by systemd.
			# By default, elxdiscoveryd does not start at time.

			# Turn off discovery daemon
			disable_elxdisco

			# Turn off auto dump
			disable_ehm_autodump

			# Turn on rmserver daemon
			enable_elxhbamgr
			stay_in_loop=0

		elif [ "$mode_of_operation" = "3" ];then
			echo "You selected: 'Remote Mode' "

			question_enable_tcpip

			if [ "$enable_tcpip_operation" = "y" ]; then
				question_management_host_address
				if [ "$management_host_address" != "" ];then
					question_exclusive_host
				fi
				question_tcp_port_number    
			fi

			# On Linux:
			# EHM daemons are controlled by systemd.
			# By default, elxdiscoveryd does not start at time.

			# Turn on discovery daemonaddress
			if [ "$platform_os" = "SunOS" ];then
				# Temporarily enable the elxdiscoveryd daemon (until the next reboot)
				# To have it start at boot, /etc/emulexDiscConfig needs to be modified,
				# which is done below.
				svcadm enable -t svc:/application/elxdiscoveryd:default > /dev/null 2>&1
			fi

			# current design has discovery daemon starting by 1st running of application
			# unless AUTOSTART entry in /etc/emulexDiscConf is true
			# /${INSTALL_DIR}/start_elxdiscovery&

			# Turn off auto dump
			disable_ehm_autodump

			# Turn on rmserver daemon
			enable_elxhbamgr
			stay_in_loop=0

		elif [ "$mode_of_operation" = "4" ];then
			echo "You selected this host as a: 'Management Host' "
			management_host_mode="y"
			question_tcp_port_number

			# On Linux:
			# EHM daemons are controlled by systemd.
			# By default, elxdiscoveryd does not start at time.

			# Turn on discovery daemonaddress
			if [ "$platform_os" = "SunOS" ];then
				# Temporarily enable the elxdiscoveryd daemon (until the next reboot)
				# To have it start at boot, /etc/emulexDiscConfig needs to be modified,
				# which is done below.
				svcadm enable -t svc:/application/elxdiscoveryd:default > /dev/null 2>&1
			fi

			# current design has discovery daemon starting by 1st running of application
			# unless AUTOSTART entry in /etc/emulexDiscConf is true
			# /${INSTALL_DIR}/start_elxdiscovery&
			# Turn off auto dump
			disable_ehm_autodump

			# Turn on rmserver daemon
			enable_elxhbamgr
			stay_in_loop=0

		elif [[ "$mode_of_operation" = "5" && "$platform_os" = "Linux" ]];then
			echo "You selected: 'Strictly Local Management with Daemons Disabled Mode' "

			daemons_enabled="n"
			disable_elxdisco
			disable_elxhbamgr
			enable_ehm_autodump
			stay_in_loop=0

		else
			stay_in_loop=1
			echo
			echo "Invalid Entry"
			if [ "$platform_os" = "Linux" ];then
				echo "Must Enter either '1', '2', '3', '4', or '5'"
			else
				echo "Must Enter either '1', '2', '3', or '4'"
			fi
		fi
	done
}

question_read_only()
{
	stay_in_loop=1
	while [ $stay_in_loop -eq 1 ]
	do
		echo ""
		echo "Would you like to enable configuration features for $PRODUCT_STRING" 
		echo "Manager clients on this platform?"
		echo ""
		echo "Enter 'y' to allow configuration (default)."
		echo "Enter 'n' for read-only mode. "
		echo
		printf "Enter the letter 'y' or 'n': "

		# Obtain user selection for Emulex HBA Manager client read only mode of operation
		read read_only_mode_of_operation
		if [ "$read_only_mode_of_operation" = "N" ] || [ "$read_only_mode_of_operation" = "n" ]; then
			read_only_mode_of_operation="y"
			echo "You selected: Read Only mode "
			stay_in_loop=0;
		elif [ "$read_only_mode_of_operation" = "Y" ] || [ "$read_only_mode_of_operation" = "y" ]; then
			read_only_mode_of_operation="n"
			echo "You selected: Yes, enable configuration"
			stay_in_loop=0;
		else
			echo "Invalid Selection "
		fi
	done
}

question_change_management_mode()
{
	# obtain user selection for change mode permission
	stay_in_loop=1
	while [ $stay_in_loop -eq 1 ]
	do
		echo ""
		echo "Do you want to allow user to change management mode using"
		if [ $FHS_SUPPORT -eq 1 ]; then
			echo "set_operating_mode script located in /${INSTALL_DIR}/scripts ?"
		else
			echo "set_operating_mode script located in /${INSTALL_DIR} ?"
		fi
		echo ""
		printf "Enter the letter 'y' if yes, or 'n' if no: "

		# Obtain user selection for Emulex HBA Manager mode of operation
		read change_mode_of_operation
		if [ "$change_mode_of_operation" = "y" ] || [ "$change_mode_of_operation" = "Y" ];then
			echo "You selected: Yes "
			stay_in_loop=0;
		fi
		if [ "$change_mode_of_operation" = "n" ] || [ "$change_mode_of_operation" = "N" ];then
			echo "You selected: No "
			stay_in_loop=0;
		fi
		if [ $stay_in_loop -eq 1 ];then
			echo "Invalid Selection "
		fi
	done
}

ocmanagerGuiCheck()
{
	# if Emulex HBA Manager GUI is running, exit script and request that user stop GUI
	#  Get pid of script that started the Emulex HBA Manager GUI 
	script_pid=`ps -eaf | grep ocmanager | grep "/bin/sh" | grep -v grep | head -n 1 | awk '{ print $2 }'`
	#  echo "script pid of hbanyware is $script_pid"
	if [ "$script_pid" != "" ];then
		gui_pid=`ps -eaf | grep java | grep -v grep | grep "OCManager.jar" | awk '{ print $2 }'`
		if [ "$gui_pid" != "" ];then
			echo "The $PRODUCT_STRING GUI client must be stopped before changing the mode of operation."
			echo "Please exit the GUI application and re-run this script"
			exit
		fi
	fi
}

callQuestionSecureManagementMode()
{
	if [ "$platform_os" = "Linux" ];then
		# On Linux, only prompt for secure management mode during installation.
		if [ $install_called -eq 0 ] || [ $install_called -eq 25 ];then
			question_secure_management_mode
		fi

		if [ "$secure_management_mode" = "s" ];then
			# Check for the four EHM groups.
			local_ocm_groups=0
			for group in "ocmlocaluser" "ocmuser" "ocmlocaladmin" "ocmadmin"; do
				found_group=`getent group | grep -ic $group`
				local_ocm_groups=`expr $local_ocm_groups + $found_group`
				if [ $found_group -eq 0 ]; then
					undefined_groups="${undefined_groups} ${group}"
				fi
			done

			# Return an error if none of the four EHM groups was found and exit script.
			if [ $local_ocm_groups -eq 0 ]; then
				echo
				echo "Error: Attempting to enable secure management feature"
				echo "       but no EHM groups are configured on this machine"
				echo "       (the output of 'getent group' command does not" 
				echo "       show any EHM groups configured)"
				echo
				echo "Reinstall without selecting EHM 'secure management' feature" 
				echo "or else configure EHM secure management groups on machine"
				echo
				exit 1
			fi
		fi    
	else
		# On Solaris, always prompt for secure management mode.
		question_secure_management_mode

		# If security has been enabled, make sure at least one of the EHM groups
		# is defined on this system.  Otherwise, proper security settings cannot 
		# cannot be applied during install.
		if [ "$secure_management_mode" = "s" ];then
			# Check for the four EHM groups.
			local_ocm_groups=0
			for group in "ocmlocaluser" "ocmuser" "ocmlocaladmin" "ocmadmin"; do
				found_group=`getent group | grep -ic $group`
				local_ocm_groups=`expr $local_ocm_groups + $found_group`
				if [ $found_group -eq 0 ]; then
					undefined_groups="${undefined_groups} ${group}"
				fi
			done

			# Return an error if none of the four EHM groups was found and exit script.
			if [ $local_ocm_groups -eq 0 ]; then
				echo ""
				echo "ERROR: Attempting to enable the Secure Management feature, but no EHM"
				echo "       groups are configured on this machine (the output of"
				echo "       'getent group' command does not show any EHM groups configured)."
				echo
				echo "Rerun without enabling the EHM 'Secure Management' feature or else"
				echo "configure EHM Secure Management groups on this machine."        
				echo ""
				exit 1
			fi
		fi
	fi
}

callQuestionManagementMode()
{
	if [ "$secure_management_mode" != "s" ];then

		question_management_mode

		# obtain user selection for 'read only' mode
		if [[ "$mode_of_operation" != "1" && "$mode_of_operation" != "4" && \
			  "$mode_of_operation" != "5" ]];then
			question_read_only
		fi

		# If called by installation routine on solaris; ask this question. Never ask on
		# Solaris since set_operating_mode is not called during installation.
		if [ $install_called -eq 45 ] || [ $install_called -eq 25 ];then
			question_change_management_mode
		fi
	fi
}

add_group_privileges_ufs()
{
	# Some files may or may not exist yet. Create them so we can set the privileges.
	touch /${INSTALL_DIR}/rm.log
	touch /${INSTALL_DIR}/cnaboardmgmt.log
	touch $RMPref

	setfacl -r -m g:ocmadmin:rwx      /${INSTALL_DIR} > /dev/null 2>&1 
	setfacl -r -m g:ocmlocaladmin:rwx /${INSTALL_DIR} > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rwx       /${INSTALL_DIR} > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rwx  /${INSTALL_DIR} > /dev/null 2>&1

	setfacl -r -m g:ocmadmin:r-x      /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:r-x /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	setfacl -r -m g:ocmuser:r-x       /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:r-x  /${INSTALL_DIR}/hbacmd > /dev/null 2>&1

	setfacl -r -m g:ocmadmin:rw-      /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rw- /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rw-       /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rw-  /${INSTALL_DIR}/rm.log > /dev/null 2>&1

	setfacl -r -m g:ocmadmin:rw-      /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rw- /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rw-       /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rw-  /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1

	# If RMRepository doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [ ! -d /${INSTALL_DIR}/RMRepository ]; then
		mkdir -p /${INSTALL_DIR}/RMRepository
	fi
	setfacl -r -m g:ocmadmin:rwx      /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rwx /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rwx       /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rwx  /${INSTALL_DIR}/RMRepository > /dev/null 2>&1

	# If misc directory doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [ ! -d /${INSTALL_DIR}/misc ]; then
		mkdir -p /${INSTALL_DIR}/misc
	fi
	setfacl -r -m g:ocmadmin:r-x      /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:r-x /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -r -m g:ocmuser:r-x       /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:r-x  /${INSTALL_DIR}/misc > /dev/null 2>&1

	# bugzill 125989
	if [ ! -r /${INSTALL_DIR}/misc/lockFile.sem ]; then
		touch /${INSTALL_DIR}/misc/lockFile.sem
	fi
	setfacl -r -m g:ocmadmin:rw-      /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rw- /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rw-       /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rw-  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1

	# bugzill 134308
	if [ ! -r /${INSTALL_DIR}/misc/hbahost_lock.sem ]; then
		touch /${INSTALL_DIR}/misc/hbahost_lock.sem
	fi   
	setfacl -r -m g:ocmadmin:rw-      /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rw- /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rw-       /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rw-  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1

	# If Dump doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [ ! -d /${INSTALL_DIR}/Dump ]; then
		mkdir -p /${INSTALL_DIR}/Dump
	fi
	setfacl -r -m g:ocmadmin:rwx      /${INSTALL_DIR}/Dump > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rwx /${INSTALL_DIR}/Dump > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rwx       /${INSTALL_DIR}/Dump > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rwx  /${INSTALL_DIR}/Dump > /dev/null 2>&1

	setfacl -r -m g:ocmadmin:rw- $DiscConfig > /dev/null 2>&1
	setfacl -r -m g:ocmadmin:rw- $RMConfig > /dev/null 2>&1
	setfacl -r -m g:ocmadmin:rw- $RMOptions > /dev/null 2>&1
	setfacl -r -m g:ocmadmin:rw- $RMPref > /dev/null 2>&1

	setfacl -r -m g:ocmlocaladmin:rw- $DiscConfig > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rw- $RMConfig > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rw- $RMOptions > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:rw- $RMPref > /dev/null 2>&1

	setfacl -r -m g:ocmuser:rw- $DiscConfig > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rw- $RMConfig > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rw- $RMOptions > /dev/null 2>&1
	setfacl -r -m g:ocmuser:rw- $RMPref > /dev/null 2>&1

	setfacl -r -m g:ocmlocaluser:rw- $DiscConfig > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rw- $RMConfig > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rw- $RMOptions > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:rw- $RMPref > /dev/null 2>&1

	setfacl -r -m g:ocmadmin:r-x      /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	setfacl -r -m g:ocmlocaladmin:r-x /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	setfacl -r -m g:ocmuser:r-x       /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	setfacl -r -m g:ocmlocaluser:r-x  /${INSTALL_DIR}/ocmanager > /dev/null 2>&1

	# Recursively set privileges to all files and subdirectories in opt/ELXocm/jre.
	find /${INSTALL_DIR}/jre -exec setfacl -r -m g:ocmadmin:r-x > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/jre -exec setfacl -r -m g:ocmlocaladmin:r-x > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/jre -exec setfacl -r -m g:ocmuser:r-x > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/jre -exec setfacl -r -m g:ocmlocaluser:r-x > /dev/null 2>&1 {} \;

	# If config doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [ ! -d /${INSTALL_DIR}/config ]; then
		mkdir -p /${INSTALL_DIR}/config
	fi
	# Recursively set privileges to all files and subdirectories in opt/ELXocm/config.
	find /${INSTALL_DIR}/config -exec setfacl -r -m g:ocmadmin:rwx > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/config -exec setfacl -r -m g:ocmlocaladmin:rwx > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/config -exec setfacl -r -m g:ocmuser:rwx > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/config -exec setfacl -r -m g:ocmlocaluser:rwx > /dev/null 2>&1 {} \;
}

remove_group_privileges_ufs()
{
	setfacl -d g:ocmadmin:      /${INSTALL_DIR} > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR} > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR} > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR} > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/hbacmd > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/rm.log > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/RMRepository > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/misc > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/Dump > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/Dump > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/Dump > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/Dump > /dev/null 2>&1

	setfacl -d g:ocmadmin: $DiscConfig > /dev/null 2>&1
	setfacl -d g:ocmadmin: $RMConfig > /dev/null 2>&1
	setfacl -d g:ocmadmin: $RMOptions > /dev/null 2>&1
	setfacl -d g:ocmadmin: $RMPref > /dev/null 2>&1

	setfacl -d g:ocmlocaladmin: $DiscConfig > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: $RMConfig > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: $RMOptions > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: $RMPref > /dev/null 2>&1

	setfacl -d g:ocmuser: $DiscConfig > /dev/null 2>&1
	setfacl -d g:ocmuser: $RMConfig > /dev/null 2>&1
	setfacl -d g:ocmuser: $RMOptions > /dev/null 2>&1
	setfacl -d g:ocmuser: $RMPref > /dev/null 2>&1

	setfacl -d g:ocmlocaluser: $DiscConfig > /dev/null 2>&1
	setfacl -d g:ocmlocaluser: $RMConfig > /dev/null 2>&1
	setfacl -d g:ocmlocaluser: $RMOptions > /dev/null 2>&1
	setfacl -d g:ocmlocaluser: $RMPref > /dev/null 2>&1

	setfacl -d g:ocmadmin:      /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	setfacl -d g:ocmlocaladmin: /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	setfacl -d g:ocmuser:       /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	setfacl -d g:ocmlocaluser:  /${INSTALL_DIR}/ocmanager > /dev/null 2>&1

	# Recursively set privileges to all files and subdirectories in opt/ELXocm/jre.
	find /${INSTALL_DIR}/jre -exec setfacl -d g:ocmadmin: > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/jre -exec setfacl -d g:ocmlocaladmin: > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/jre -exec setfacl -d g:ocmuser: > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/jre -exec setfacl -d g:ocmlocaluser: > /dev/null 2>&1 {} \;

	# Recursively set privileges to all files and subdirectories in opt/ELXocm/config.
	find /${INSTALL_DIR}/config -exec setfacl -d g:ocmadmin: > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/config -exec setfacl -d g:ocmlocaladmin: > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/config -exec setfacl -d g:ocmuser: > /dev/null 2>&1 {} \;
	find /${INSTALL_DIR}/config -exec setfacl -d g:ocmlocaluser: > /dev/null 2>&1 {} \;
}

add_group_privileges_zfs()
{
	# Some files may or may not exist yet. Create them so we can set the privileges.
	touch /${INSTALL_DIR}/rm.log
	touch /${INSTALL_DIR}/cnaboardmgmt.log
	touch $RMPref

	/usr/bin/chmod  A+group:ocmadmin:rwx:allow       /${INSTALL_DIR} > /dev/null 2>&1 
	/usr/bin/chmod  A+group:ocmlocaladmin:rwx:allow  /${INSTALL_DIR} > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rwx:allow        /${INSTALL_DIR} > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rwx:allow   /${INSTALL_DIR} > /dev/null 2>&1

	/usr/bin/chmod  A+group:ocmadmin:rx:allow       /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rx:allow  /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rx:allow        /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rx:allow   /${INSTALL_DIR}/hbacmd > /dev/null 2>&1

	/usr/bin/chmod  A+group:ocmadmin:rw:allow       /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rw:allow  /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rw:allow        /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rw:allow   /${INSTALL_DIR}/rm.log > /dev/null 2>&1

	/usr/bin/chmod  A+group:ocmadmin:rw:allow       /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rw:allow  /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rw:allow        /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rw:allow   /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1

	# If RMRepository doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [ ! -d /${INSTALL_DIR}/RMRepository ]; then
		mkdir -p /${INSTALL_DIR}/RMRepository
	fi
	/usr/bin/chmod  A+group:ocmadmin:rwx:allow       /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rwx:allow  /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rwx:allow        /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rwx:allow   /${INSTALL_DIR}/RMRepository > /dev/null 2>&1

	# If misc directory doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [ ! -d /${INSTALL_DIR}/misc ]; then
		mkdir -p /${INSTALL_DIR}/misc
	fi
	/usr/bin/chmod  A+group:ocmadmin:rx:allow /${INSTALL_DIR}/misc > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rx:allow /${INSTALL_DIR}/misc > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rx:allow /${INSTALL_DIR}/misc > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rx:allow /${INSTALL_DIR}/misc > /dev/null 2>&1

	# bugzilla 125989
	if [ ! -r /${INSTALL_DIR}/misc/lockFile.sem ]; then
		touch /${INSTALL_DIR}/misc/lockFile.sem
	fi   
	/usr/bin/chmod  A+group:ocmadmin:rw:allow  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1

	# bugzilla 134308
	if [ ! -r /${INSTALL_DIR}/misc/hbahost_lock.sem ]; then
		touch /${INSTALL_DIR}/misc/hbahost_lock.sem
	fi   
	/usr/bin/chmod  A+group:ocmadmin:rw:allow  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1

	# If Dump doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [ ! -d /${INSTALL_DIR}/Dump ]; then
		mkdir -p /${INSTALL_DIR}/Dump
	fi
	/usr/bin/chmod  A+group:ocmadmin:rwx:allow       /${INSTALL_DIR}/Dump > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rwx:allow  /${INSTALL_DIR}/Dump > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rwx:allow        /${INSTALL_DIR}/Dump > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rwx:allow   /${INSTALL_DIR}/Dump > /dev/null 2>&1

	/usr/bin/chmod  A+group:ocmadmin:rw:allow $DiscConfig > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow $RMConfig > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow $RMOptions > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmadmin:rw:allow $RMPref > /dev/null 2>&1

	/usr/bin/chmod  A+group:ocmlocaladmin:rw:allow $DiscConfig > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rw:allow $RMConfig > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rw:allow $RMOptions > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rw:allow $RMPref > /dev/null 2>&1

	/usr/bin/chmod  A+group:ocmuser:rw:allow $DiscConfig > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rw:allow $RMConfig > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rw:allow $RMOptions > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmuser:rw:allow $RMPref > /dev/null 2>&1

	/usr/bin/chmod  A+group:ocmlocaluser:rw:allow $DiscConfig > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rw:allow $RMConfig > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rw:allow $RMOptions > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rw:allow $RMPref > /dev/null 2>&1

	/usr/bin/chmod  A+group:ocmadmin:rx:allow       /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaladmin:rx:allow  /${INSTALL_DIR}/ocmanager > /dev/null 2>&1 
	/usr/bin/chmod  A+group:ocmuser:rx:allow        /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	/usr/bin/chmod  A+group:ocmlocaluser:rx:allow   /${INSTALL_DIR}/ocmanager > /dev/null 2>&1

	/usr/bin/chmod -R  A+group:ocmadmin:rx:allow      /${INSTALL_DIR}/jre > /dev/null 2>&1
	/usr/bin/chmod -R  A+group:ocmlocaladmin:rx:allow /${INSTALL_DIR}/jre > /dev/null 2>&1
	/usr/bin/chmod -R  A+group:ocmuser:rx:allow       /${INSTALL_DIR}/jre > /dev/null 2>&1
	/usr/bin/chmod -R  A+group:ocmlocaluser:rx:allow  /${INSTALL_DIR}/jre > /dev/null 2>&1

	# If config doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [ ! -d /${INSTALL_DIR}/config ]; then
		mkdir -p /${INSTALL_DIR}/config
	fi
	/usr/bin/chmod -R A+group:ocmadmin:rw:allow       /${INSTALL_DIR}/config > /dev/null 2>&1
	/usr/bin/chmod -R A+group:ocmlocaladmin:rw:allow  /${INSTALL_DIR}/config > /dev/null 2>&1
	/usr/bin/chmod -R A+group:ocmuser:rw:allow        /${INSTALL_DIR}/config > /dev/null 2>&1
	/usr/bin/chmod -R A+group:ocmlocaluser:rw:allow   /${INSTALL_DIR}/config > /dev/null 2>&1
}

remove_group_privileges_zfs()
{
	/usr/bin/chmod -f A-  /${INSTALL_DIR} > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/hbacmd > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/rm.log > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/cnaboardmgmt.log > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/RMRepository > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/misc > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/Dump > /dev/null 2>&1
	/usr/bin/chmod -f A-  $RMPref > /dev/null 2>&1
	/usr/bin/chmod -f A-  $RMPref > /dev/null 2>&1
	/usr/bin/chmod -f A-  $RMPref > /dev/null 2>&1
	/usr/bin/chmod -f A-  $RMPref > /dev/null 2>&1
	/usr/bin/chmod -f A-  /${INSTALL_DIR}/ocmanager > /dev/null 2>&1
	/usr/bin/chmod -fR A- /${INSTALL_DIR}/jre > /dev/null 2>&1	
	/usr/bin/chmod -fR A- /${INSTALL_DIR}/config > /dev/null 2>&1	
}

add_group_privileges_linux()
{
	# Some files may or may not exist yet. Create them so we can set the privileges.
	# This file should not be here!
	touch $RMPref

	setfacl -m g:ocmadmin:rwx      /${INSTALL_DIR}
	setfacl -m g:ocmlocaladmin:rwx /${INSTALL_DIR}
	setfacl -m g:ocmuser:rwx       /${INSTALL_DIR}
	setfacl -m g:ocmlocaluser:rwx  /${INSTALL_DIR}

	if [[ $FHS_SUPPORT -eq 1 ]]; then
		if [[ -d /${BIN_DIR} ]]; then
			setfacl -m g:ocmadmin:rwx      /${BIN_DIR}
			setfacl -m g:ocmlocaladmin:rwx /${BIN_DIR}
			setfacl -m g:ocmuser:rwx       /${BIN_DIR}
			setfacl -m g:ocmlocaluser:rwx  /${BIN_DIR}

			setfacl -m g:ocmadmin:rx       /${BIN_DIR}/*
			setfacl -m g:ocmlocaladmin:rx  /${BIN_DIR}/*
			setfacl -m g:ocmuser:rx        /${BIN_DIR}/*
			setfacl -m g:ocmlocaluser:rx   /${BIN_DIR}/*
		fi

		if [[ -d /${INSTALL_DIR}/scripts ]]; then
			setfacl -m g:ocmadmin:rwx      /${INSTALL_DIR}/scripts
			setfacl -m g:ocmlocaladmin:rwx /${INSTALL_DIR}/scripts
			setfacl -m g:ocmuser:rwx       /${INSTALL_DIR}/scripts
			setfacl -m g:ocmlocaluser:rwx  /${INSTALL_DIR}/scripts

			setfacl -m g:ocmadmin:rx       /${INSTALL_DIR}/scripts/*.sh
			setfacl -m g:ocmlocaladmin:rx  /${INSTALL_DIR}/scripts/*.sh
			setfacl -m g:ocmuser:rx        /${INSTALL_DIR}/scripts/*.sh
			setfacl -m g:ocmlocaluser:rx   /${INSTALL_DIR}/scripts/*.sh
		fi

		if [[ -d /${INSTALL_DIR}/scripts/lpfc ]]; then
			setfacl -m g:ocmadmin:rwx      /${INSTALL_DIR}/scripts/lpfc
			setfacl -m g:ocmlocaladmin:rwx /${INSTALL_DIR}/scripts/lpfc
			setfacl -m g:ocmuser:rwx       /${INSTALL_DIR}/scripts/lpfc
			setfacl -m g:ocmlocaluser:rwx  /${INSTALL_DIR}/scripts/lpfc

			setfacl -m g:ocmadmin:rx       /${INSTALL_DIR}/scripts/lpfc/*.sh
			setfacl -m g:ocmlocaladmin:rx  /${INSTALL_DIR}/scripts/lpfc/*.sh
			setfacl -m g:ocmuser:rx        /${INSTALL_DIR}/scripts/lpfc/*.sh
			setfacl -m g:ocmlocaluser:rx   /${INSTALL_DIR}/scripts/lpfc/*.sh
		fi

		if [[ -d /${INSTALL_DIR}/gui ]]; then
			setfacl -R -m g:ocmadmin:rx      /${INSTALL_DIR}/gui
			setfacl -R -m g:ocmlocaladmin:rx /${INSTALL_DIR}/gui
			setfacl -R -m g:ocmuser:rx       /${INSTALL_DIR}/gui
			setfacl -R -m g:ocmlocaluser:rx  /${INSTALL_DIR}/gui
		fi
	else
		setfacl -m g:ocmadmin:rx        $CLI_BIN
		setfacl -m g:ocmlocaladmin:rx   $CLI_BIN
		setfacl -m g:ocmuser:rx         $CLI_BIN
		setfacl -m g:ocmlocaluser:rx    $CLI_BIN

		setfacl -m g:ocmadmin:rx        $CONVERT_CONF_SH
		setfacl -m g:ocmlocaladmin:rx   $CONVERT_CONF_SH 
		setfacl -m g:ocmuser:rx         $CONVERT_CONF_SH
		setfacl -m g:ocmlocaluser:rx    $CONVERT_CONF_SH
	fi

	setfacl -m g:ocmadmin:rwx       /${INSTALL_DIR}/RMRepository
	setfacl -m g:ocmlocaladmin:rwx  /${INSTALL_DIR}/RMRepository
	setfacl -m g:ocmuser:rwx        /${INSTALL_DIR}/RMRepository
	setfacl -m g:ocmlocaluser:rwx   /${INSTALL_DIR}/RMRepository

	# [Bug 151562] - Update OCM to comply with FHS
	setfacl -m g:ocmadmin:rw        $RMConf
	setfacl -m g:ocmlocaladmin:rw   $RMConf
	setfacl -m g:ocmuser:rw         $RMConf
	setfacl -m g:ocmlocaluser:rw    $RMConf

	setfacl -m g:ocmadmin:rw        /${LOG_DIR}/cnaboardmgmt.log
	setfacl -m g:ocmlocaladmin:rw   /${LOG_DIR}/cnaboardmgmt.log
	setfacl -m g:ocmuser:rw         /${LOG_DIR}/cnaboardmgmt.log
	setfacl -m g:ocmlocaluser:rw    /${LOG_DIR}/cnaboardmgmt.log

	setfacl -R -m g:ocmadmin:rwx      /${DUMP_DIR}
	setfacl -R -m g:ocmlocaladmin:rwx /${DUMP_DIR}
	setfacl -R -m g:ocmuser:rwx       /${DUMP_DIR}
	setfacl -R -m g:ocmlocaluser:rwx  /${DUMP_DIR}

	setfacl -m g:ocmadmin:rw        /${LOG_DIR}/featuremgmt.log
	setfacl -m g:ocmlocaladmin:rw   /${LOG_DIR}/featuremgmt.log
	setfacl -m g:ocmuser:rw         /${LOG_DIR}/featuremgmt.log
	setfacl -m g:ocmlocaluser:rw    /${LOG_DIR}/featuremgmt.log

	setfacl -m g:ocmadmin:rw        /${LOG_DIR}/rm.log
	setfacl -m g:ocmlocaladmin:rw   /${LOG_DIR}/rm.log
	setfacl -m g:ocmuser:rw         /${LOG_DIR}/rm.log
	setfacl -m g:ocmlocaluser:rw    /${LOG_DIR}/rm.log

	setfacl -m g:ocmadmin:rw        /${LOG_DIR}/wwnmgmt.log
	setfacl -m g:ocmlocaladmin:rw   /${LOG_DIR}/wwnmgmt.log
	setfacl -m g:ocmuser:rw         /${LOG_DIR}/wwnmgmt.log
	setfacl -m g:ocmlocaluser:rw    /${LOG_DIR}/wwnmgmt.log

	# If misc directory doesn't exist yet, create it so we can ensure that
	# the proper privileges are set.
	if [[ ! -d /${INSTALL_DIR}/misc ]]; then
		mkdir -p /${INSTALL_DIR}/misc
	fi
	setfacl -m g:ocmadmin:rx      /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -m g:ocmlocaladmin:rx /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -m g:ocmuser:rx       /${INSTALL_DIR}/misc > /dev/null 2>&1
	setfacl -m g:ocmlocaluser:rx  /${INSTALL_DIR}/misc > /dev/null 2>&1

	# bugzilla 133450
	if [[ ! -r /${INSTALL_DIR}/misc/lockFile.sem ]]; then
		touch /${INSTALL_DIR}/misc/lockFile.sem
	fi   
	setfacl -m g:ocmadmin:rw      /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaladmin:rw /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -m g:ocmuser:rw       /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaluser:rw  /${INSTALL_DIR}/misc/lockFile.sem > /dev/null 2>&1

	# bugzilla 134308
	if [[ ! -r /${INSTALL_DIR}/misc/hbahost_lock.sem ]]; then
		touch /${INSTALL_DIR}/misc/hbahost_lock.sem
	fi   
	setfacl -m g:ocmadmin:rw      /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaladmin:rw /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmuser:rw       /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaluser:rw  /${INSTALL_DIR}/misc/hbahost_lock.sem > /dev/null 2>&1

	# bugzilla 251272
	if [[ ! -r /${INSTALL_DIR}/misc/hbacimhost_lock.sem ]]; then
		touch /${INSTALL_DIR}/misc/hbacimhost_lock.sem
	fi
	setfacl -m g:ocmadmin:rw      /${INSTALL_DIR}/misc/hbacimhost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaladmin:rw /${INSTALL_DIR}/misc/hbacimhost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmuser:rw       /${INSTALL_DIR}/misc/hbacimhost_lock.sem > /dev/null 2>&1
	setfacl -m g:ocmlocaluser:rw  /${INSTALL_DIR}/misc/hbacimhost_lock.sem > /dev/null 2>&1

	setfacl -m g:ocmadmin:rw $DiscConfig
	setfacl -m g:ocmadmin:rw $RMConfig
	setfacl -m g:ocmadmin:rw $RMOptions
	setfacl -m g:ocmadmin:rw $RMPref

	setfacl -m g:ocmlocaladmin:rw $DiscConfig
	setfacl -m g:ocmlocaladmin:rw $RMConfig
	setfacl -m g:ocmlocaladmin:rw $RMOptions
	setfacl -m g:ocmlocaladmin:rw $RMPref

	setfacl -m g:ocmuser:rw $DiscConfig
	setfacl -m g:ocmuser:rw $RMConfig
	setfacl -m g:ocmuser:rw $RMOptions
	setfacl -m g:ocmuser:rw $RMPref

	setfacl -m g:ocmlocaluser:rw $DiscConfig
	setfacl -m g:ocmlocaluser:rw $RMConfig
	setfacl -m g:ocmlocaluser:rw $RMOptions
	setfacl -m g:ocmlocaluser:rw $RMPref

	if [[ -r $GUI_BIN ]]; then
		setfacl -m g:ocmadmin:rx      $GUI_BIN
		setfacl -m g:ocmlocaladmin:rx $GUI_BIN
		setfacl -m g:ocmuser:rx       $GUI_BIN
		setfacl -m g:ocmlocaluser:rx  $GUI_BIN
	fi

	if [[ -d /${INSTALL_DIR}/jre ]]; then
		setfacl -R -m g:ocmadmin:rx      /${INSTALL_DIR}/jre
		setfacl -R -m g:ocmlocaladmin:rx /${INSTALL_DIR}/jre
		setfacl -R -m g:ocmuser:rx       /${INSTALL_DIR}/jre
		setfacl -R -m g:ocmlocaluser:rx  /${INSTALL_DIR}/jre
	fi

	if [[ -d /${INSTALL_DIR}/config ]]; then
		setfacl -m g:ocmadmin:rwx       /${INSTALL_DIR}/config
		setfacl -m g:ocmlocaladmin:rwx  /${INSTALL_DIR}/config
		setfacl -m g:ocmuser:rwx        /${INSTALL_DIR}/config
		setfacl -m g:ocmlocaluser:rwx   /${INSTALL_DIR}/config

		setfacl -m g:ocmadmin:rw        /${INSTALL_DIR}/config/*
		setfacl -m g:ocmlocaladmin:rw   /${INSTALL_DIR}/config/*
		setfacl -m g:ocmuser:rw         /${INSTALL_DIR}/config/*
		setfacl -m g:ocmlocaluser:rw    /${INSTALL_DIR}/config/*    
	fi
}

remove_group_privileges_linux()
{
	if [[ $FHS_SUPPORT -eq 1 ]]; then
		setfacl -R -b /${CONF_DIR}
		setfacl -R -b /${INSTALL_DIR}
		setfacl -R -b /${VAR_DIR}
		return
	fi

	setfacl -b /${INSTALL_DIR}
	setfacl -b ${CLI_BIN}
	setfacl -b ${CONVERT_CONF_SH} 
	setfacl -b /${INSTALL_DIR}/RMRepository
	setfacl -b $DiscConfig
	setfacl -b $RMConfig
	setfacl -b $RMConf
	setfacl -b $RMOptions
	setfacl -b $RMPref

	if [[ -d /${INSTALL_DIR}/scripts ]]; then
		setfacl -R -b /${INSTALL_DIR}/scripts
	fi

	if [[ -d /${INSTALL_DIR}/misc ]]; then
		setfacl -R -b /${INSTALL_DIR}/misc
	fi

	if [[ -d /${DUMP_DIR} ]]; then
		setfacl -R -b /${DUMP_DIR}
	fi

	if [[ -d /${LOG_DIR} ]]; then
		setfacl -R -b /${LOG_DIR}
	fi

	if [[ -d /${RM_CONF_DIR} ]]; then
		setfacl -R -b /${RM_CONF_DIR}
	fi

	if [[ -r ${GUI_BIN} ]]; then
		setfacl -b ${GUI_BIN}
	fi

	if [[ -d /${INSTALL_DIR}/jre ]]; then
		setfacl -R -b /${INSTALL_DIR}/jre
	fi

	if [[ -d /${INSTALL_DIR}/config ]]; then
		setfacl -b /${INSTALL_DIR}/config
		setfacl -b /${INSTALL_DIR}/config/*
	fi
}

setFilePermissionsSolaris()
{
	# Add or remove group privileges based on Secure Management selection.
	if [ "$platform_os" = "SunOS" ];then
		if [ "$secure_management_mode" = "s" ];then
			# Secure Management has been enabled. Add group privileges on the necessary files.
			if [ `/usr/bin/df -n | grep -w / | awk '{ print $3 }'` = "ufs" ]; then 
				# UFS filesystem
				add_group_privileges_ufs
			else
				# ZFS filesystem
				add_group_privileges_zfs
			fi

			# Warn the user that if any of the 4 EHM groups are not currently defined, they
			# will not be given access privileges and if the group is added later, it will
			# not have privileges. 
			if [ "$undefined_groups" != "" ]; then
				echo ""
				echo "WARNING: The following EHM groups are not currently defined on this system"
				echo "and could not be granted access privileges. If these groups are added at a"
				echo "later time, run /${INSTALL_DIR}/set_operating_mode or reinstall to grant"
				echo "them privileges:"
				echo ""
				for group in $undefined_groups; do
					echo "  ${group}"
				done
			fi
		else
			# Secure Management has been disabled. Remove group privileges on the necessary files.    
			if [ `/usr/bin/df -n | grep -w / | awk '{ print $3 }'` = "ufs" ]; then 
				# UFS filesystem
				remove_group_privileges_ufs
			else
				# ZFS filesystem
				remove_group_privileges_zfs
			fi
		fi
	fi
}

setFilePermissionsLinux()
{
	if [ "$platform_os" = "Linux" ];then
		if [ $secure_management_mode = "s" ];then
			# Secure Management has been enabled. Add group privileges on the necessary files.
			if [ $install_called -eq 0 ];then
				# if called directly by user, set file permission
				add_group_privileges_linux > /dev/null 2>&1
				# Warn the user that if any of the 4 EHM groups are not currently defined, they
				# will not be given access privileges and if the group is added later, it will
				# not have privileges. 
				if [ "$undefined_groups" != "" ]; then
					echo ""
					echo "WARNING: The following EHM groups are not currently defined on this system"
					echo "and could not be granted access privileges. If these groups are added at a"
					echo "later time, run /${INSTALL_DIR}/set_operating_mode or reinstall to grant"
					echo "them privileges:"
					echo ""
					for group in $undefined_groups; do
						echo "  ${group}"
					done
				fi  
			fi
		else
			# Secure Management has been disabled. Remove group privileges on the necessary files.
			if [ $install_called -eq 0 ];then
				# if called directly by user, set file permission
				remove_group_privileges_linux > /dev/null 2>&1
			fi
		fi
	fi
}

write_configuration_to_file()
{
	# write server remote operation option to config file  
	TF=/${CONF_DIR}/emulexRMConfig.tmp
	cat $RMConfig | sed "/RemoteOperation/d" > $TF
	if [[ $mode_of_operation -eq 1 ||  $mode_of_operation -eq 5 ]];then
		echo "ServerRemoteOperation:disabled" >> $TF
	else
		echo "ServerRemoteOperation:enabled" >> $TF
	fi
	cp $TF /${CONF_DIR}/emulexRMConfig
	rm $TF

	# write autostart setting config file
	if [ $mode_of_operation -eq 5 ];then
		TF=/${CONF_DIR}/emulexDiscConfig.tmp
		cat $DiscConfig | sed "/AUTOSTART/d" > $TF
		echo "AUTOSTART:false" >> $TF
		cp $TF $DiscConfig
		rm $TF
	fi

	# write client remote operation option to config file  
	TF=/${CONF_DIR}/emulexDiscConfig.tmp
	cat $DiscConfig | sed "/RemoteOperation/d" > $TF
	if [ $mode_of_operation -eq 3 ] || [ $mode_of_operation -eq 4 ];then
		echo "ClientRemoteOperation:enabled" >> $TF
	else
		echo "ClientRemoteOperation:disabled" >> $TF
	fi
	cp $TF $DiscConfig
	rm $TF

	# this information only processed when called by install routine
	if [ $install_called -eq 45 ] || [ $install_called -eq 25 ];then
		TF=/${CONF_DIR}/emulexDiscConfig.tmp
		cat $DiscConfig | sed "/OperationMode/d" > $TF
		if [ "$change_mode_of_operation" = "y" ];then
			echo "ClientOperationMode:enabled" >> $TF
		else
			echo "ClientOperationMode:locked" >> $TF
		fi
		cp $TF $DiscConfig
		rm $TF
	else
		echo "ClientOperationMode:enabled" >> $TF    
	fi

	# init related flag to same selection
	# commenting out next line as part of fix for CR 29557
	#if [ "$change_mode_of_operation" = "Y" ];then
	TF=/${CONF_DIR}/emulexRMOptions.tmp
	cat $RMOptions | sed "/LocalHBAsOnly/d" > $TF
	if [ $mode_of_operation -eq 3 ] || [ $mode_of_operation -eq 4 ];then
		echo "LocalHBAsOnly:false" >> $TF
	else
		echo "LocalHBAsOnly:true" >> $TF
	fi
	cp $TF $RMOptions
	rm $TF

	# place user's 'read only' configuration selection into config file
	TF=/${CONF_DIR}/emulexDiscConfig.tmp
	cat $DiscConfig | sed "/ClientReadOnlyOperation/d" > $TF
	if [ $read_only_mode_of_operation = "y" ];then    
		echo "ClientReadOnlyOperation:enabled" >> $TF
	else
		echo "ClientReadOnlyOperation:disabled" >> $TF
	fi
	cp $TF $DiscConfig
	rm $TF

	# place 'tcp/ip socket' port number into config file
	TF=/${CONF_DIR}/emulexDiscConfig.tmp
	cat $DiscConfig | sed "/TcpSocketPortNumber/d" > $TF
	echo "TcpSocketPortNumber:$socket_port_value" >> $TF
	cp $TF $DiscConfig
	rm $TF

	# place 'ManagementHost' value into config file
	TF=/${CONF_DIR}/emulexRMConfig.tmp
	cat $RMConfig | sed "/ManagementHost/d" > $TF
	if [ $management_host_mode = "y" ];then
		echo "ManagementHost:true" >> $TF
	else
		echo "ManagementHost:false" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF

	# place 'SecureManagement' value into config file
	TF=/${CONF_DIR}/emulexRMConfig.tmp
	cat $RMConfig | sed "/SecureManagement/d" > $TF
	if [ $secure_management_mode = "s" ];then
		echo "SecureManagement:enabled" >> $TF
	else
		echo "SecureManagement:disabled" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF

	# place management host name or ip address value into config file
	TF=/${CONF_DIR}/emulexRMConfig.tmp
	cat $RMConfig | sed "/MngmtHostIpAddress/d" > $TF
	echo "MngmtHostIpAddress:$management_host_address" >> $TF
	cp $TF $RMConfig
	rm $TF

	# place exclusive mgmnt host boolean into config file
	TF=/${CONF_DIR}/emulexRMConfig.tmp
	cat $RMConfig | sed "/ExclusiveHostMgmnt/d" > $TF
	if [ $management_host_is_exclusive = "y" ];then
		echo "ExclusiveHostMgmnt:true" >> $TF
	else
		echo "ExclusiveHostMgmnt:false" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF

	# place 'DaemonsEnabled' boolean into config file
	TF=/${CONF_DIR}/emulexRMConfig.tmp
	cat $RMConfig | sed "/DaemonsEnabled/d" > $TF
	if [ $daemons_enabled = "y" ];then
		echo "DaemonsEnabled:true" >> $TF
	else
		echo "DaemonsEnabled:false" >> $TF
	fi
	cp $TF $RMConfig
	rm $TF
}

stop_ocmanager()
{
	if [ -x "${STOP_EHM_SH}" ]; then
		${STOP_EHM_SH} > /dev/null 2>&1
	fi
}

start_ocmanager()
{
	if [ -x "${START_EHM_SH}" ]; then
		${START_EHM_SH} > /dev/null 2>&1
	fi
}

setDirPaths
ocmanagerGuiCheck

# first stop two daemons
echo "Stopping $PRODUCT_STRING daemons ..."
stop_ocmanager

callQuestionSecureManagementMode
callQuestionManagementMode
setFilePermissionsSolaris
setFilePermissionsLinux
write_configuration_to_file

# added next lines as part of fix for CR 29557
# discovery daemon was intermittently not reporting to gui when 'local only'
# or 'managed only' modes were selected, restarting daemon fixed problem
echo
stop_ocmanager
echo

# remote server daemon is started (for 'dump' and such) unless
# management mode 5 was selected
if [ $daemons_enabled == "y" ]; then
	start_ocmanager
fi

